import java.util.Iterator;
import java.util.NoSuchElementException;

/*
 * This file shows how to define an iterator in a single Java source file using
 * nested classes (specifically, inner classes).  Observe that the `firstNode`
 * field is no longer required, since the outer instance provides a persistent
 * reference to the head node.
 */

/** Node in a potentially circular linked list. */
public class CNodeOuter<T> implements Iterable<T> {
    private T value;
    private CNodeOuter<T> next;

    /** `next` may be null. */
    public CNodeOuter(T value, CNodeOuter<T> next) {
        this.value = value;
        this.next = next;
    }

    /** Returns an iterator over elements in the list, starting with this node's value.
     *  Precondition: list is linear, or this node is on circular portion. */
    @Override
    public Iterator<T> iterator() {
        return new CNodeIterator();
    }

    /**
     * Iterator over elements in a linear or circular linked list with outer
     * object as head.
     */
    private class CNodeIterator implements Iterator<T> {
        /** Node whose data will be returned on the next call to `next()`. */
        private CNodeOuter<T> nextNode;

        /** True if `next()` has been called at least once. */
        private boolean hasYielded;

        /** Create a new iterator starting at the outer node instance. */
        public CNodeIterator() {
            nextNode = CNodeOuter.this;
            hasYielded = false;
        }

        @Override
        public boolean hasNext() {
            // Deliberately not using `equals() here - we expect to return to
            // the same node object we started from.
            return nextNode != null && (nextNode != CNodeOuter.this || !hasYielded);
        }

        @Override
        public T next() throws NoSuchElementException {
            if (!hasNext()) {
                throw new NoSuchElementException();
            }
            T ans = nextNode.value;
            nextNode = nextNode.next;
            hasYielded = true;
            return ans;
        }
    }

    /** Demo: Iterate over a circular linked list with 4 nodes. */
    public static void main(String[] args) {
        CNodeOuter<String> n4 = new CNodeOuter<>("E", null);
        CNodeOuter<String> n3 = new CNodeOuter<>("F", n4);
        CNodeOuter<String> n2 = new CNodeOuter<>("A", n3);
        CNodeOuter<String> n1 = new CNodeOuter<>("C", n2);
        n4.next = n1;

        // Enhanced for-loop uses CNode.iterator()
        for (String s : n1) {
            System.out.print(s + " ");
        }
        System.out.println();
    }
}
