import java.util.Arrays;

/** Contains static method sort to sort an int array using heapsort.
 *  Also contains static methods to manipulate int max-heaps in an array.
 *  They are public, for testing purposes. But if you want this class just
 *  to show procedure sortheap, make all the other methods private. */
public class Heapsort {
    
    /** README: Procedure heapsort comes first. It has two parts,
     * (1) Make its parameter b into a max-heap.
     * (2) Extract elements from the heap, one by one, from largest to smallest,
     *     and put them in the right place.
     * 
     * In order to do this, a max-heap and not a min-heap is used. This means
     * that every value in the heap is >= its children. We write "max-heap"
     * instead of "heap" everywhere below as a reminder.
     * 
     * The max-heap invariant for array segment b[0..n-1] is given 
     * after procedure heapsort.
     * 
     * Then follows all the methods for heaps shown in class: insert, extract,
     * bubbleUp, and bubbleDown. We do this only for an array of ints and not
     * generically to keep it simple, for best understanding of what is going on.
     */
    
    /** Sort b. */
    public static void heapsort(int[] b) {
        // Make b into a max-heap
        // invariant: b[0..k-1] is a max-heap  AND
        //            b is a permutation of its initial value
        for (int k= 0; k < b.length; k= k+1) {
            add(b[k], b, k);
        }
        
        // Sort the max-heap
        // invariant: b[0..k-1] is a max-heap AND
        //            b[0..k-1] <= b[k..] AND
        //            b[k..] is sorted AND
        //            b is a permutation of its initial value
        for (int k= b.length; k > 0; k= k-1) {
            int temp= b[0];
            poll(b, k);
            b[k-1]= temp;
        }
    }

    /** Max-heap invariant for b[0..n-1]:
        b[0..n-1] is viewed as a max-heap, i.e.
        1. All elements b[0..n-1] are filled with a value
        2. the children of b[i] are b[2i+1] and b[2i+2]
        3. the parent of b[i] is b[(i-1)/2]
        4. the children of a node b[i] are <= b[i]
     */

    /** Add v to max-heap b[0..n-1] and return the new size.
     *  Precondition: b[0..n-1] is a max-heap.*/
    public static int add(int v, int[] b, int n) {
        b[n]= v;
        bubbleUp(b, n);
        return n+1;
    }

    /** Remove the maximum element and return the size of the heap.
        Precondition: b[0..n-1] is a max-heap.
        If n = 0, throw IllegalArgumentException.*/
    public static int poll(int b[], int n) {
        if (n == 0) throw new IllegalArgumentException();
        n= n-1;
        b[0]= b[n];
        bubbleDown(b, n);
        return n;
    }

    /** Max-Heapify b[0..n] by bubbling b[n] up.
     * Precondition: b[0..n-1] is a max-heap */
    public static void bubbleUp(int[] b, int n) {
        int k= n;
        int p= (k-1)/2;   // p is the parent of k!

        // invariant: Every element satisfies the max-heap property
        // except perhaps b[k] might be larger than its parent.
        // p is the parent of k.
        while (k > 0  &&  b[k] > b[p]) {
            int temp= b[k]; b[k]= b[p]; b[p]= temp;
            k= p;
            p= (k-1)/2;
        }
    }

    /** Max-Heapify b[0..n-1] by bubble b[0] down to its heap position.!
     * Precondition: b[0..n-1] is a max-heap except that the root
     * may have a larger child */
    public static void bubbleDown(int[] b, int n) {
        int k= 0;
        // Set c to larger of b[k]'s children
        int c= 2*k + 2;     // b[k]'s right child
        if (c > n-1 || b[c-1] > b[c])  c= c-1;

        // inv; tree is a max-heap except: b[k] may have a larger a child.
        //        Also b[k]'s largest child is b[c]
        while (c < n  &&  b[k] < b[c]) {
            int temp= b[k]; b[k]= b[c]; b[c]= temp;
            k= c;
            c= 2*k + 2;      // b[k]'s right child!
            if (c > n-1 || b[c-1] > b[c])  c= c-1;
        }
    }
}