
import java.io.*;

/** Static methods to parse an expression and generate code for it */
public class ExpressionCompiler {
    /** This class has static methods for parsing an arithmetic expression
     * containing ints, +, -, *, and / and parentheses ( ) and generating
     * postfix code for it. This postfix code has one instruction per
     * line and is similar to machine language.
     * 
     * The grammar for expressions is this:
     * 
     * <Exp>    := <Term>   {<+ or -> <Term>}
     * <Term>   := <Factor> {<* or /> <Factor>}
     * <Factor> := integer  |   - <Factor>    |  ( <Expr> )
     */

    /** Read lines from the keyboard until an empty line is read; then output DONE.
     *  Parse each line as an expression and output the postfix code for it
     */
    public static void main(String[] args) throws IOException {
        // Create and store in kbd a keyboard reader
        System.out.println("Starting main program");
        BufferedReader kbd=  new BufferedReader(new InputStreamReader(System.in));
        
        String line= kbd.readLine();
        while (line.length() > 0) {
            Scanner scanner= new Scanner(line);
            try { 
                String output= parseExp(scanner);
                System.out.println(output);
            } catch (Error r) {
                System.out.println(r);
            }

            line= kbd.readLine();
        }
        System.out.println("Empty keyboard line; program terminated");

    }

    /**  scanner's input should start with an <Exp> --if not
     *             throw a RuntimeException.
     *   Return the corresponding postfix instructions
     *   and have scanner remove the <Exp> from its input.
     *   
     *   An <Exp> is a <Term> followed by any number of + or -  <Term>
     *   
     *   <Exp> := <Term> {<+ or -> <Term>}
     */
    public static String parseExp(Scanner scanner) {
        String code= parseTerm(scanner);
        while  ("+".equals(scanner.token()) || "-".equals(scanner.token())) {
            String operator= scanner.scanOverToken();
            String rightOp= parseTerm(scanner);
            code=  code + rightOp + (operator.equals("+") ? "PLUS\n" : "MINUS\n");
        }

        return code;
    }
    
    /**  scanner's input should start with a <Term>
     *       --if not, throw a RuntimeException.
     *   Return the corresponding postfix instructions,
     *   and have scanner remove the <Term> from its string.
     *   
     *   A <Term> is a <Factor> followed by any number of: (* or /)  <Factor>
     *   
     *   <Term> := <Factor> {<* or /> <Factor>}
     */
    public static String parseTerm(Scanner scanner) {
        String code= parseFactor(scanner);
        while  ("*".equals(scanner.token()) || "/".equals(scanner.token())) {
            String operator= scanner.scanOverToken();
            String rightOp= parseFactor(scanner);
            code=  code + rightOp + ("*".equals(operator) ? "MULT\n" : "DIV\n");      
        }

        return code;
    }

    /**  scanner's input should start with a <Factor> if not, throw
     *   a RuntimeException.
     *   Return the postfix instructions for <Factor>
     *   and have scanner remove the <Factor> from its input.
     *   
     *   <Factor> := an integer  |
     *             - <Factor>    |
     *             (  <Expr> )
     */
    public static String parseFactor(Scanner scanner) {
        if (scanner.tokenIsInt() ) {
            String code= "PUSH " + scanner.token() + "\n";
            scanner.scanOverToken();
            return code;
        }

        if (scanner.token().equals("-")) {
            scanner.scanOverToken();          // scan over the -
            String code= parseFactor(scanner);
            return code + "NEG\n";
        }

        // <Factor> should be ( <Expr> )
        getToken(scanner, "(");

        String code= parseExp(scanner);

        getToken(scanner, ")");

        return code;
    }

    /** If scanner's next token matches tok, remove it from the scanner
     *  and return that token.
     *  If not, throw RuntimeException with appropriate error message.
     */
    public static String getToken(Scanner scanner, String tok) {
        if (scanner.token() == null) throw new RuntimeException("Expected more input");
        String token= scanner.token();
        if (tok.equals(token)) {
            scanner.scanOverToken();
            return tok;
        }
        throw new RuntimeException("Expected " + tok + " but found " + token);
    }

}
