/** An instance provides tokens from string, one at a time.
 * A token is either
     * 1. an unsigned integer,
     * 2. a Java identifier
     * 3. an operator + - * / %
     * 4. a paren of some sort: ( ) [ ] { }
     * 5. any sequence of non-whitespace chars not included in 1..4. */
public class Scanner {

    /** The unprocessed input consists of the token in token
     * followed by input.
     * token = null means that there are no more tokens --i.e. also input is empty
     * Neither token nor input has beginning/trailing whitespace
     */
    private String token; 
    private String input; 

    private static final String oneCharTokens= "+-*%()[]{}/";

    /** An instance that provides the token for s */
    public Scanner(String s) {
        input= s.trim();
        putNextToken();
    }

    /** Return true iff s has more unprocessed tokens */
    public boolean hasToken() {
        return token != null;
    }

    /** Return the first unprocessed token null if none */
    public String token() {
        return token;
    }
    
    /** Remove the first unprocessed token (thus processing it)
     *   and return it. */
    public String scanOverToken() {
        String t= token;
        putNextToken();
        return t;    
    }

    /** Return true iff first unprocessed token is an int */
    public boolean tokenIsInt() {
        return token != null && Character.isDigit(token.charAt(0));
    }

    /** Return true iff first unprocessed token is an identifier
     * i.e. a letter followed by 0 or more letters/digits */
    public boolean tokenIsId() {
        return token != null && Character.isJavaIdentifierStart(token.charAt(0));
    }

    /** move the next token from input to token --store null
        in token if none. */
    public void putNextToken() {
        if (input.length() == 0) {
            token= null;
            return;
        }
        char firstChar= input.charAt(0);
        token= "";
        if (Character.isDigit(firstChar)) {
           putInt();
           return;
        }
        if (Character.isJavaIdentifierStart(firstChar)) {
            putId();
            return;
        }
        if (oneCharTokens.contains(firstChar + "")) {
            token= firstChar + "";
            input= input.substring(1).trim();
            return;
        }
    }

    /** input starts with a digit. Put the first token in token and
     * remove it from input.*/
    private void putInt() {
        // Set k to the index of first char of input that is not a digit
        int k= 1;
        while (k < input.length() && Character.isDigit(input.charAt(k))) {
            k= k+1;
        }
        token= input.substring(0,k);
        input= input.substring(k).trim();    
    }
    
    /** input starts a Java identifier. Put the first token in token and
     * remove it from input.*/
    private void putId() {
        // Set k to the index of first char of input that is not an identifier char
        int k= 1;
        while (k < input.length() && Character.isJavaIdentifierPart(input.charAt(k))) {
            k= k+1;
        }
        token= input.substring(0,k);
        input= input.substring(k).trim();    
    }
    
    /** input starts a token that is not an int, an identifier, an operator,
     *  or a paren of some kind. Put the token in token and remove it from input .*/
    private void putWeird() {
        // Set k to the index of first char of input that is a char in kind 1..4
        int k= 1;
        while (k < input.length() &&
                !Character.isDigit(input.charAt(k)) &&
                !Character.isJavaIdentifierStart(input.charAt(k)) &&
                !Character.isJavaIdentifierPart(input.charAt(k)) &&
                !oneCharTokens.contains(input.charAt(k) + "")  &&
                !Character.isWhitespace(input.charAt(k))) {
            k= k+1;
        }
        token= input.substring(0,k).trim();
        input= input.substring(k).trim();    
    }
    
}
