import java.util.ArrayList;
import java.util.Arrays;

public class NaiveHashTable {
	class Pair {
		Pair(String k, String v) { key = k; value = v; }
		String key;
		String value;
		public String toString() { return key+":"+value; }
	}
	
	int size; // number of elements
	int numBuckets; // number of buckets
	int k; // upper bound on elements/buckets
	       // when exceeded, numBuckets is doubled, elements are rehashed
	ArrayList<ArrayList<Pair>> buckets;
	
	public NaiveHashTable(int initNumBuckets, int k) {
		this.size = 0;
		numBuckets = initNumBuckets;
		this.k = k;
		
		buckets = new ArrayList<ArrayList<Pair>>(numBuckets);
		for (int i=0; i<numBuckets; i++) {
			buckets.add(new ArrayList<Pair>());
		}
	}
	
	// Returns the value for the key,
	// null if there is no entry with this key.
	public String get(String key) {
		int hash = key.hashCode() % numBuckets;
		ArrayList<Pair> list = buckets.get(hash);
		for (Pair p: list) {
			if (p.key.equals(key)) return p.value;
		}
		return null;
	}
	
	// Returns the previous value of the specified key in the
	// hashtable, or null if it did not have one. 
	public String put(String key, String value) {
		int hash = key.hashCode() % numBuckets;
		ArrayList<Pair> list = buckets.get(hash);
		for (Pair p: list) {
			if (p.key.equals(key)) {
				// replace the value
				p.value = value;
				return value;
			}
		}
		
		// key not found, have to add a new pair
		if (size+1 > k*numBuckets) {
			doubleAndRehash();
			hash = key.hashCode() % numBuckets;
		}
		buckets.get(hash).add(new Pair(key,value));
		size++;
		
		return value;
	}
	
	private void doubleAndRehash() {
		int newNumBuckets = 2*numBuckets;
		ArrayList<ArrayList<Pair>> newBuckets = new ArrayList<ArrayList<Pair>>(newNumBuckets);
		for (int i=0; i<newNumBuckets; i++) {
			newBuckets.add(new ArrayList<Pair>());
		}
		
		// rehash
		for (ArrayList<Pair> list: buckets) {
			for (Pair p: list) {
				int hash = p.key.hashCode() % newNumBuckets;
				newBuckets.get(hash).add(p);
			}
		}
		
		numBuckets = newNumBuckets;
		buckets = newBuckets;
	}
	
	public void print() {
		System.out.println("# buckets = " + numBuckets);
		System.out.println("# elements = " + size + " (bound = " + (numBuckets*k) + ")");
		for (int i=0; i<numBuckets; i++) {
			System.out.print("bucket " + i + ": [");
			ArrayList<Pair> list = buckets.get(i);
			int n = list.size();
			if (n > 0) System.out.print(list.get(0));
			for (int j=1; j<n; j++) System.out.print(", " + list.get(j));
			System.out.println("]");
		}
	}
	
	public static void main(String[] args) {
		NaiveHashTable table = new NaiveHashTable(2,3);
		table.print();
		for (int i=0; i<30; i++) {
			table.put(""+i,"v"+i);
			table.print();
		}
		String[] keys = new String[]{ "10", "100" };
		for (String key: keys) 
			System.out.println(table.get(key));
	}
	
}
