/* This is an incomplete hash table implementation.  In particular, it provides
 * no provision for extending the hash table when it hits capacity.  This is a
 * simple "key" store, analogous to a set.  One could extend the table to also
 * implement a "key-value" store, but that sullies the point.
 *
 * This particular hash table uses the method of separate chaining[1] to
 * illustrate the manner in which keys are mapped onto buckets.
 *
 * 1.  http://en.wikipedia.org/wiki/Hash_table#Separate_chaining
 */

import java.util.ArrayList;

public class HashTable
{
    private HashFunc m_hf;
    private ArrayList<ArrayList<String> > m_table;

    /** Create a hash table of static size which uses hf
     *
     * The hash table will allow an arbitrary number of items to be inserted;
     * however, if the load factor gets sufficiently high, searching degrades
     * toward O(n).  I wouldn't use this hash table in production, but it serves
     * as a good illustration of how hash functions work.
     */
    public HashTable(HashFunc hf, int numBuckets)
    {
        m_hf = hf;
        m_table = new ArrayList<ArrayList<String> >(numBuckets);

        for (int i = 0; i < numBuckets; ++i)
        {
            m_table.add(new ArrayList<String>());
        }
    }

    /** Insert String s into the hash table.
     */
    public void add(String s)
    {
        int position = m_hf.map(s);

        if (position < 0)
        {
            position = 0 - position;
        }
        if (position < 0) // This takes care of when the above actually overflows.
        {
            position = 0;
        }

        m_table.get(position % m_table.size()).add(s);
    }

    /** Show how full each bucket of the hash table is.
     */
    public void showDistribution()
    {
        System.out.println("Table buckets are filled to the following capacities:");

        for (int i = 0; i < m_table.size(); ++i)
        {
            System.out.print(m_table.get(i).size() + " ");
        }

        System.out.println("");
    }

    /** Show which words fall into each bucket.
     */
    public void dumpTable()
    {
        System.out.println("Table buckets contain the following strings:");

        for (int i = 0; i < m_table.size(); ++i)
        {
            boolean printed = false;

            for (int j = 0; j < m_table.get(i).size(); ++j)
            {
                System.out.print(m_table.get(i).get(j) + " ");
                printed = true;
            }

            if (printed)
            {
                System.out.println("");
            }
        }

        System.out.println("");
    }
}
