package cs2110.netID.phylogeny.util;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import cs2110.netID.phylogeny.*;

/**
 * Takes a .dat file and builds an animal from it.
 */
public class DatParser {
	
	private static final Pattern LINE_PATTERN = Pattern.compile("\\s*(\\w+)\\s*=\\s*\"([^\"]+)\"\\s*");

	/**
	 * Parses a single line, which has the form Attribute="Value"
	 * 
	 * @param line - line to be parsed
	 * @return array of [attribute, value]
	 */
	private static String[] parseLine(String line) {
		Matcher matcher = LINE_PATTERN.matcher(line);
		if (!matcher.matches()) {
			String message = String.format("Invalid line format: %s", line);
			throw new IllegalArgumentException(message);
		}
		return new String[] { matcher.group(1), matcher.group(2) };
	}
	
	/**
	 * Builds an animal from a .dat file.
	 * 
	 * @param filename - name of .dat file to parse
	 * @return animal parsed from the file
	 * @throws IOException
	 */
	public static Species parseAnimal(String filename) throws IOException {
		return parseAnimal(new File(filename));
	}
	
	/**
	 * Builds an animal from a .dat file.
	 * 
	 * @param file - File object to parse
	 * @return animal parsed from the file
	 * @throws IOException
	 */
	public static Species parseAnimal(File file) throws IOException {
		String name = "";
		String latinName = "";
		String imageSource = "";
		String dna = "";
		
		// Open the file for parsing.
		BufferedReader reader = new BufferedReader(new FileReader(file));
		while (true) {
			String line = reader.readLine();
			
			// Done reading.
			if (line == null) {
				break;
			}
			
			// Skip blank lines.
			if (line.trim().equals("")) {
				continue;
			}
			
			// Get attribute and value.
			String[] parts = parseLine(line);
			String attribute = parts[0];
			String value = parts[1];
			
			// Fill the appropriate field.
			if (attribute.equals("Name")) {
				name = value;
			}
			else if (attribute.equals("LatinName")) {
				latinName = value;
			}
			else if (attribute.equals("ImageFilename")) {
				imageSource = value;
			}
			else if (attribute.equals("DNA")) {
				dna = value;
			}
			else {
				String message = String.format("Unexpected attribute: %s", attribute);
				reader.close();
				throw new IllegalArgumentException(message);
			}
		}
		// Construct the animal.
		reader.close();
		return new Species(name, latinName, imageSource, dna);
	}
	
}