package cs2110.netID.phylogeny.test;

import java.io.File;
import java.io.IOException;
import java.util.HashSet;
import java.util.Set;

import org.junit.*;

import cs2110.netID.phylogeny.*;
import cs2110.netID.phylogeny.util.*;


public abstract class PhylogenyTests {
	
	protected static final int NUM_ANIMALS = 40;
	protected static int ROOT_ANIMAL = 17;
	
	protected static Set<Species> animals;
	protected static Species rootAnimal;

	/**
	 * Loads all 40 animals from .dat files with A17 as the root.
	 * 
	 * @throws IOException 
	 */
	@BeforeClass
	public static void createAnimals() throws IOException {
		animals = new HashSet<Species>();
		rootAnimal = null;
		for (int i = 0; i < NUM_ANIMALS; i ++) {
			String filename = String.format("A%d.dat", i);
			File file = new File(filename);
			if (!file.exists()) {
				Assert.fail(String.format("Count not find %s; do you have the right working directory?", filename));
			}
			
			Species animal = DatParser.parseAnimal(filename);
			animals.add(animal);
			if (i == ROOT_ANIMAL) {
				rootAnimal = animal;
			}
		}
	}
	
	/**
	 * Checks the tree has no long paths or cycles.
	 * We don't know if a long path is a cycle,
	 * but we do know that cycles will cause an infinite loop.
	 * 
	 * @param tree - tree to check
	 * @param size - number of nodes in the tree
	 */
	protected void assertNoLongPathsOrCycles(PhylogenyTree tree, int size) {
		longPathOrCycleFinder(tree, size, 0);
	}
	
	/**
	 * Looks for a long path or cycle recursively.
	 * 
	 * @param tree - tree to check
	 * @param size - expected number of nodes in the tree
	 * @param depth - current depth into the tree
	 */
	private void longPathOrCycleFinder(PhylogenyTree tree, int size, int depth) {
		if (depth >= size) {
			Assert.fail(String.format("Tree should have %d nodes, but it has either a cycle or a path with more than %d nodes.", size, size));
		}
		for (PhylogenyTree child : tree.getChildren()) {
			longPathOrCycleFinder(child, size, depth + 1);
		}
	}
	
	/**
	 * Builds a phylogeny tree from the pre-built animals and checks its against potential cycles.
	 * 
	 * @return phylogeny tree
	 */
	protected PhylogenyTree getPhylogenyTree() {
		PhylogenyTree tree = new PhylogenyTree(animals, rootAnimal);
		assertNoLongPathsOrCycles(tree, NUM_ANIMALS);
		return tree;
	}

}
