/**********************************************************************************
 This class demonstrates a graphic version of puzzle.
 ************************************************************************************/

import java.awt.event.*;
import java.awt.*;
import javax.swing.*;
import java.util.Stack;
import java.io.*;

public class NPuzzle extends JFrame implements ActionListener {

   private int[][] puzzle; // the int array representation of the puzzle
   private int size; // the size of the puzzle
   private int blankRow, blankCol; // position of the blank
   private int numMove; // the number of moves you have done
   private JButton[][] tiles; // the JButton representation of the puzzle
   private Stack<String> moveStack; // the stack to store the moves
   private TileListener[][] listeners; // the listeners responding to the tile
   // buttons
   private JLabel message; // the place to display message
   private JPanel board; // the panel containing all the tile buttons
   private Timer scheduler; // schedules the steps in the animation of solve

   public NPuzzle() {
      this(3);
   }

   public NPuzzle(int size) {
      initBoard(size);
      initFrame();
   }

   // Initialize the array puzzle and the panel board containing the tiles
   private void initBoard(int size) {
      this.size = size;
      numMove = 0;
      moveStack = new Stack<String>();

      // initialize puzzle
      puzzle = new int[size][size];
      int count = 1;
      for (int row = 0; row < size; row++)
         for (int col = 0; col < size; col++)
            puzzle[row][col] = count++;

      blankRow = size - 1;
      blankCol = size - 1;
      puzzle[blankRow][blankCol] = 0;

      // add tiles to the board
      board = new JPanel();
      board.setLayout(new GridLayout(size, size));

      tiles = new JButton[size][size];
      listeners = new TileListener[size][size];

      for (int i = 0; i < size; i++)
         for (int j = 0; j < size; j++) {
            tiles[i][j] = new JButton("" + puzzle[i][j]);
            tiles[i][j].setBackground(Color.yellow);
            listeners[i][j] = new TileListener(i, j);
            tiles[i][j].addActionListener(listeners[i][j]);
            // adjust the size of the button according to number of buttons in
            // the puzzle
            tiles[i][j].setMargin(new Insets(200 / size, 200 / size,
                  200 / size, 200 / size));
            board.add(tiles[i][j]);
         }

      JButton blank = tiles[blankRow][blankCol];
      blank.setText("");
      blank.setBackground(Color.white);
   }

   // Initialize the buttons and the message object, put everything in the frame
   private void initFrame() {
      JPanel buttonPanel = new JPanel();
      buttonPanel.setLayout(new GridLayout(7, 1, 20, 20));

      JButton info = new JButton("Info");
      info.addActionListener(this);
      buttonPanel.add(info);

      JButton file = new JButton("File");
      file.addActionListener(this);
      buttonPanel.add(file);

      JButton scramble = new JButton("Scramble");
      scramble.addActionListener(this);
      buttonPanel.add(scramble);

      JButton solve = new JButton("Solve");
      solve.addActionListener(this);
      buttonPanel.add(solve);

      JButton interrupt = new JButton("Interrupt");
      interrupt.addActionListener(this);
      buttonPanel.add(interrupt);

      JButton resize = new JButton("Resize");
      resize.addActionListener(this);
      buttonPanel.add(resize);

      JButton quit = new JButton("Quit");
      quit.addActionListener(this);
      buttonPanel.add(quit);

      message = new JLabel();
      message.setFont(new Font("Tahoma", Font.PLAIN, 24));
      message.setText("Welcome to play puzzle.");
      message.setHorizontalAlignment(JLabel.CENTER);

      Container container = getContentPane();
      container.setLayout(new BorderLayout(20, 10));

      container.add(board, BorderLayout.CENTER);
      container.add(buttonPanel, BorderLayout.WEST);
      container.add(message, BorderLayout.SOUTH);
   }

   // Respond to the buttons in buttonPanel
   public void actionPerformed(ActionEvent e) {
      String command = e.getActionCommand();

      message.setText("");
      if (command.equals("Info")) {
         String content = "This is a demo. of the CS211 puzzle game. ";
         content += "It uses JFrame to implement the frame, JPanel to implement board and button panels, and Jbutton to implement tiles and buttons";
         content += "The messages are displayed with JLabel class.";
         content += "The layout of tiles is managed by GridLayout. The layout of the message, board and button panels is managed by BorderLayout.";
         JTextArea text = new JTextArea(content);
         text.setLineWrap(true);
         text.setWrapStyleWord(true);
         text.setFont(new Font("Tahoma", Font.PLAIN, 20));
         JScrollPane scroll = new JScrollPane(text);
         scroll.setPreferredSize(new Dimension(300, 300));
         JOptionPane.showMessageDialog(null, scroll);
      } else if (command.equals("File")) {
         String fileName = JOptionPane
         .showInputDialog("Enter the file name you want to upload to move the puzzle");
         if (fileName != null) readMovesFile(fileName);
      } else if (command.equals("Scramble")) {
         String input = JOptionPane
         .showInputDialog("Enter the number of times to scramble");
         int nScramble = Integer.parseInt(input);
         scramble(nScramble);
      } else if (command.equals("Solve")) {
         scheduler = new Timer(1000, new SolvePerformer());
         scheduler.start();
      } else if (command.equals("Interrupt")) {
         scheduler.stop();
      } else if (command.equals("Resize")) {
         String input = JOptionPane.showInputDialog("Enter the new size");
         int size = Integer.parseInt(input);
         if (size < 2 || size > 32) {
            message.setText("size must be between 2 and 32");
            return;
         }
         Container container = getContentPane();
         container.remove(board);
         initBoard(size);
         container.add(board, BorderLayout.CENTER);
         pack();
      } else if (command.equals("Quit")) {
         System.exit(0);
      }
   }

   public static void main(String[] args) {

      NPuzzle frame;

      // If number of arguments is less than 1, use the default size.
      // Otherwise, read the size from the first argument.
      if (args.length >= 1) {
         int size = Integer.parseInt(args[0]);
         frame = new NPuzzle(size);
      } else {
         frame = new NPuzzle();
      }

      frame.setTitle("puzzle game");
      frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
      frame.pack();
      frame.setVisible(true);

   }

   // Return true if puzzle is solved:
   public boolean isSolved() {
      // Are values inside the array OK?
      int count = 1;
      for (int row = 0; row < size; row++)
         for (int col = 0; col < size; col++) {
            if (puzzle[row][col] != count) {
               if (row != size - 1 || col != size - 1 || puzzle[row][col] != 0)
                  return false;
            }
            count++;
         }
      return true;
   }

   // Executes a sequence of moves as specified in a file
   // A sample file might contain "E\nS\nW\nN\nE\nS\nW\nS\nE\nE\nN", where each
   // letter
   // represents a direction to slide a tile.
   public void readMovesFile(String filename) {
      String direction = null;
      BufferedReader inFile;

      try {
         inFile = new BufferedReader(new FileReader(filename));
         try {
            direction = inFile.readLine();
            while (direction != null) {
               int destRow = -1;
               int destCol = -1;

               if (direction.equalsIgnoreCase("N")) {
                  destRow = blankRow + 1;
                  destCol = blankCol;
               } else if (direction.equalsIgnoreCase("S")) {
                  destRow = blankRow - 1;
                  destCol = blankCol;
               } else if (direction.equalsIgnoreCase("E")) {
                  destRow = blankRow;
                  destCol = blankCol - 1;
               } else if (direction.equalsIgnoreCase("W")) {
                  destRow = blankRow;
                  destCol = blankCol + 1;
               }

               if (!attemptMove(destRow, destCol))
                  message.setText("File has specified an illegal move!");
               direction = inFile.readLine();
            }
         } catch (IOException e) {
            message.setText("File read went wrong!");
         }

         try {
            // Close the file!
            inFile.close();
         } catch (IOException e) {}

      } catch (FileNotFoundException e) {
         message.setText("The input file does not exist!");
      }
   }

   // scramble the puzzle numScramble times and push the moves onto moveStack
   // Update blankRow and blankCol
   // Suggest: only keep and count valid moves and store them onto the stack
   public void scramble(int numScramble) {
      boolean successfulMove;
      int randRow, randCol;

      // repeat numScramble times
      for (int i = 0; i < numScramble; i++) {

         // repeat until a successful move has been made
         successfulMove = false;
         while (!successfulMove) {
            // pick random tile to move
            randRow = (int) Math.floor(Math.random() * size);
            randCol = (int) Math.floor(Math.random() * size);

            // attempt the move
            successfulMove = attemptMove(randRow, randCol, true);
         }

      }
      numMove = 0;
   }

   // check to see if the destination for the blank is a valid move.
   // If yes, change the puzzle accordingly, increment numMove and push the
   // move onto moveStack, update blankRow and blankCol, return true.
   // If not, return false.
   public boolean attemptMove(int destinationRow, int destinationCol) {
      return attemptMove(destinationRow, destinationCol, false);
   }

   // private helper method
   // if prohibitRetrace is true, don't allow consecutive moves by the same tile
   private boolean attemptMove(int destinationRow, int destinationCol,
         boolean prohibitRetrace) {
      // check if tile is within bounds
      boolean outOfBounds = (destinationRow < 0 || destinationRow >= size
            || destinationCol < 0 || destinationCol >= size);
      if (outOfBounds) return false;

      // check if chosen tile is adjacent to blank
      boolean above = destinationRow == blankRow - 1;
      boolean below = destinationRow == blankRow + 1;
      boolean left = destinationCol == blankCol - 1;
      boolean right = destinationCol == blankCol + 1;
      boolean sameCol = blankCol == destinationCol;
      boolean sameRow = blankRow == destinationRow;
      boolean adjacentToBlank = (((left || right) && sameRow) || ((above || below) && sameCol));
      if (!adjacentToBlank) return false;

      // determine which direction to slide tile
      String direction = null;
      if (above) direction = "SOUTH";
      else if (below) direction = "NORTH";
      else if (left) direction = "EAST";
      else if (right) direction = "WEST";
      else {
         message.setText("Error");
         return false;
      }

      // if prohibitRetrace flag is on,
      // check that same tile is not being moved back and forth consecutively
      // (we can determine the last move by peeking at the top of the stack)
      if (prohibitRetrace && !moveStack.empty()) {
         String lastMove = moveStack.peek();
         if ((direction == "NORTH" && lastMove == "SOUTH")
               || (direction == "SOUTH" && lastMove == "NORTH")
               || (direction == "EAST" && lastMove == "WEST")
               || (direction == "WEST" && lastMove == "EAST")) return false;
      }

      // move tile
      moveTile(destinationRow, destinationCol);

      // push move onto stack
      moveStack.push(direction);

      // increment numMove
      numMove++;

      return true;
   }

   // Do the actual move of the tile/blank.
   private void moveTile(int destinationRow, int destinationCol) {
      // update puzzle accordingly
      puzzle[blankRow][blankCol] = puzzle[destinationRow][destinationCol];
      puzzle[destinationRow][destinationCol] = 0;

      // update the tiles accordingly
      tiles[blankRow][blankCol].setText("" + puzzle[blankRow][blankCol]);
      tiles[destinationRow][destinationCol].setText("");
      tiles[blankRow][blankCol].setBackground(Color.yellow);
      tiles[destinationRow][destinationCol].setBackground(Color.white);

      // update blank row/col (set equal to randRow/randCol)
      blankRow = destinationRow;
      blankCol = destinationCol;
   }

   // Perform the task of solve the puzzle by reversing the moves on the
   // moveStack
   class SolvePerformer implements ActionListener {

      public void actionPerformed(ActionEvent e) {
         if (moveStack.empty()) {
            scheduler.stop();
            if (isSolved()) message.setText("The puzzle is now solved");
            return;
         }

         String direction = moveStack.pop();

         // find out how to swap tiles. no switches on strings allowed
         // do opposite of direction because made direction to get there need to
         // do opposite to undo it

         // move a tile south into blank location
         if (direction.equalsIgnoreCase("NORTH")) moveTile(blankRow - 1,
               blankCol);

         // move a tile north into blank location
         else if (direction.equalsIgnoreCase("SOUTH")) moveTile(blankRow + 1,
               blankCol);

         // move tile east into blank location
         else if (direction.equalsIgnoreCase("WEST")) moveTile(blankRow,
               blankCol - 1);

         // move tile west into blank location
         else if (direction.equalsIgnoreCase("EAST"))
            moveTile(blankRow, blankCol + 1);
      }
   }

   //respond to the tiles and try to move accordingly.
   class TileListener implements ActionListener {

      private int row, column;

      TileListener(int row, int column) {
         this.row = row;
         this.column = column;
      }

      public void actionPerformed(ActionEvent e) {
         if (attemptMove(row, column)) {
            if (isSolved()) message.setText("You solved the puzzle in "
                  + numMove + " steps.");
            else message.setText("You have moved " + numMove + " steps");
         } else {
            message.setText("Illegal moves!");
         }
      }
   }
}
