package gui;

import game.Constants;
import game.Game;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.InputStream;

import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.Clip;
import javax.sound.sampled.LineEvent;
import javax.sound.sampled.LineListener;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.UIManager;

/**
 * Class to create the GUI for the Game
 */
public class GUI extends JFrame implements Constants {

	private static final long serialVersionUID = 4322838670943725248L;
	private final Animator animator = new Animator();
	private final Game game;
	private final JButton startButton = new JButton("Start");
	private final Background bg;

	public final Sound openingSong = new Sound("openingSong");
	public final Sound eatDot = new Sound("eatdot");
	public final Sound eatBigDot = new Sound("eatBigDot2");
	public final Sound die = new Sound("die");
	public final Sound eatingGhosts = new Sound("eatingGhosts");
	public final Sound intermission = new Sound("intermission");
   
   private static boolean silent = false; // silent mode
   
   public static void main(String[] args) {
      silent = args.length > 0 && args[0].equals("-s");
      try {
         UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
      } catch (Exception e) {}
      new GUI();
   }

	public GUI() {
		setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		setTitle("CS211 Pacman");

		game = new Game(this);
		bg = new Background(game);

		//layout components
		add(bg, BorderLayout.CENTER);

		JPanel controlPanel = new JPanel();
		controlPanel.add(startButton);
		add(controlPanel, BorderLayout.SOUTH);

		//initialize listeners
		startButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				if (startButton.getText().equals("Start")) {
					openingSong.playAndWait(10);
					startButton.setText("Pause");
					GUI.this.requestFocusInWindow();
					game.playGame();
					animator.start();
				} else if (startButton.getText().equals("Pause")) {
					animator.stopAnimation();
					startButton.setText("Resume");
				} else { //button text = "Resume"
					startButton.setText("Pause");
					GUI.this.requestFocusInWindow();
					restartAnimation();
				}
			}
		});
		startButton.setEnabled(true);

		addKeyListener(game.localPlayer);

		pack();
		Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
		setLocation((screenSize.width - getWidth())/2, (screenSize.height - getHeight())/2);
		setVisible(true);
	}

	public void updateLocations(double dt) {
		game.updateUnits(game.localPlayer, dt);
		game.updateUnits(game.aiPlayer, dt);
	}

	public void stopAnimation() {
		animator.stopAnimation();
	}

	public void restartAnimation() {
		synchronized(animator) {
			animator.notify();
		}
	}

	/**
	 * Class for running Animations.
	 */
	class Animator extends Thread {
		final long FRAME_TIME = 60; // min # of milliseconds/frame
		final double DT_MAX = .03; // max frame latency
		boolean isRunning = false; // Is the simulator running?

		// start the simulation
		public synchronized void run() {
			double dt = 0; // delta time (change in time from frame to frame)
			long nextFrame = 0; // when the next frame should occur
			long lastFrame = System.currentTimeMillis(); // the time of the last frame
			long firstFrame = lastFrame;

			nextFrame = System.currentTimeMillis() + FRAME_TIME;
			isRunning = true; // Is the simulator running?
			long now = 0;

			// main game loop
			while (true) {
				while (isRunning) {
					now = System.currentTimeMillis();

					// incremental update
					if (now > nextFrame) {
						// update the locations of everything
						updateLocations(FRAME_TIME/1000.0);

						game.checkCollisions();
						// redraw to the screen
						bg.repaint();

						// update the time
						lastFrame = now;
						nextFrame = now + FRAME_TIME;
					}

					// calculate the delta time and scale if running slowly
					dt = (double) (now - lastFrame) / 1000;
					if (dt > DT_MAX) dt = DT_MAX;
					bg.setAnimationTime(now - firstFrame);
				}

				//someone stopped us
				try {
					wait();         
				} catch (InterruptedException ie) {}
				isRunning = true;
			}
		}

		public void stopAnimation() {
			isRunning = false;
		}

		public boolean isRunning() {
			return isRunning;
		}
	}

	/**
	 * Sounds
	 */
	public class Sound implements LineListener {
		private Clip clip;

		Sound(String soundFileName) {
			InputStream sound = ClassLoader.getSystemResourceAsStream(soundDirectory + soundFileName + ".wav");
			try {
				clip = AudioSystem.getClip();
				clip.open(AudioSystem.getAudioInputStream(sound));
				clip.addLineListener(this);
			} catch (Exception e) {
				clip = null;
			}
		}

		public void play() {
			if (silent || clip == null) return;
			clip.setFramePosition(0);
			clip.loop(0);
		}

		//wait the specified number of seconds, or until
		//the song is over, whichever comes first
		public synchronized void playAndWait(int seconds) {
			if (silent || clip == null) return;
			clip.setFramePosition(0);
			clip.loop(0);
			try {
				wait(seconds * 1000);
			} catch (InterruptedException ie) {}
		}

		//to implement the LineListener interface
		public synchronized void update(LineEvent le) {
			if (le.getType().equals(LineEvent.Type.STOP)) notify();
		}
	}
}
