package gui;

import game.Constants;
import game.Game;
import game.Phase;
import game.actor.Ghost;
import game.actor.Pacman;
import game.map.Cell;
import game.map.PacmanMap;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.image.BufferedImage;
import java.util.List;

import javax.swing.JPanel;

/**
 * Class to extend the functionality of ScrollablePicture so that we can create
 * a background image from a map definition file. This class also serves as the
 * primary redering platform - all on-screen drawing occurs in the
 * paintComponent method
 * 
 * @author Alec Berntson
 * 
 */
public class Background extends JPanel implements Constants {

	private static final long serialVersionUID = -1567109021282882103L;

	private static final String SPLASH_SCREEN_STRING = "Splash Screen...";

	// Reference to the game that this is being played on
	protected Game game;

	// keep track of time left in the animation phase
	protected long animationTime = 0;

	BufferedImage backgroundImage;
	Cell[][] cellGrid;

	/**
	 * Create a new background for game. The scrollable image and listeners are
	 * created.
	 * 
	 * @param game
	 *           reference to the game object. this is used for getting units to
	 *           draw and statistics
	 */
	public Background(Game game) {
		this.game = game;
		PacmanMap map = game.map;
		cellGrid = map.getCellGrid();
		backgroundImage = map.getBackground();
		setPreferredSize(new Dimension(CELL*map.getTileWidth(), CELL*map.getTileHeight()));
	}

	/**
	 * This method is called automatically by java whenever repaint() is called.
	 * All drawing to the screen occurs here
	 * 
	 * @param g
	 *           the graphics object for all drawing commands that java provides
	 *           when this method is called
	 */
	public void paintComponent(Graphics g) {
		super.paintComponent(g);

		// paint the board
		g.drawImage(backgroundImage, 0, 0, null);

		//display the splash screen
		if (game.getPhase() == Phase.SPLASH_SCREEN) {
			g.setColor(Color.RED);
			g.drawString(SPLASH_SCREEN_STRING, (int) g.getClipBounds().getCenterX(), (int) g.getClipBounds().getCenterY());
		}

		//draw the navigable cells
		game.drawTokens(g);

		if (game.getPhase() == Phase.PLAY_GAME) {
			Pacman pacman = game.localPlayer.getPacman();
			pacman.draw(g);
			List<Ghost> ghostList = game.aiPlayer.getActorList();

			for (Ghost gh : ghostList) {
				gh.draw(g);
			}

			g.drawString(" t: " + (int) (animationTime / 1000) + ":" + (int) ((animationTime / 10) % 100), (int) g.getClipBounds().getMinX() + 200, (int) g.getClipBounds().getMinY() + 10);
		}

	}

	/**
	 * returns the time remaining for the animation phase
	 * 
	 * @return long the animation time
	 */
	public long getAnimationTime() {
		return animationTime;
	}

	/**
	 * sets the time remaining for the animation phase
	 * 
	 * @param animationTime
	 *           the new time remaining
	 */
	public void setAnimationTime(long animationTime) {
		this.animationTime = animationTime;
	}
}
