package game;

import game.actor.Actor;
import game.actor.Ghost;
import game.actor.Pacman;
import game.actor.Token;
import game.map.Cell;
import game.map.PacmanMap;
import game.player.AIPlayer;
import game.player.LocalPlayer;
import game.player.Player;
import gui.GUI;

import java.awt.Graphics;
import java.awt.geom.Point2D;
import java.util.HashSet;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;

/**
 * This is the 'Central' class that holds all the data and state for the game.
 * It holds on to the local and remote player, the game phase, the map, and the
 * various resource tables. 
 */
public class Game implements Constants {

	// the player that controls the ghosts
	public final AIPlayer aiPlayer;
	// the player who has control of the GUI
	public final LocalPlayer localPlayer;
	/**
	 * This is the graph you have to write. 
	 */
	public final PacmanMap map;
	// the gui
	public final GUI gui;
	// which phase of the game was are in
	private Phase phase;

	private Set<Token> tokenList = new HashSet<Token>();
	private Set<Actor> actorList = new HashSet<Actor>();
	private Timer vulnerableGhostTimer;

	/**
	 * Constructor for the game.
	 * 
	 * @param gui
	 *           reference to the GUI
	 */
	public Game(GUI gui) {
		this.gui = gui;

		map = new PacmanMap();
		Actor.setGame(this);

		/*
		 * add the token objects
		 */
		Cell[][] cells = map.getCellGrid();
		for (int y = 0; y < map.getTileHeight(); y++) {
			for (int x = 0; x < map.getTileWidth(); x++) {
				if (cells[x][y].isToken()) {
					char type = cells[x][y].type;
					Token t = new Token(new Point2D.Double(x * CELL, y * CELL), type == '*');
					tokenList.add(t);
				}
			}
		}

		// create the players
		localPlayer = new LocalPlayer(this);
		aiPlayer = new AIPlayer(this, map);
		actorList.add(localPlayer.getPacman());
		actorList.addAll(aiPlayer.getActorList());

		phase = Phase.SPLASH_SCREEN;
	}

	public void playGame() {
		phase = Phase.PLAY_GAME;
	}

	/**
	 * Update the location of all of players' units based on dt amount of time
	 * passing.
	 * 
	 * @param player
	 *           the player where we getUnits() from
	 * @param dt
	 *           the amount of times (in seconds) that we need to move the unit
	 *           over. Actor speeds are given in pixels per second, so dt*speed
	 *           is the movement potential
	 */
	public void updateUnits(Player player, double dt) {
		player.move(dt);
	}
	/**
	 * draw all the tokens to the screen
	 * @param g
	 */
	public void drawTokens(Graphics g) {
		for (Token t : tokenList) {
			t.draw(g);
		}
	}
	/**
	 * check if pacman has hit any of the ghosts or tokens.
	 * Take the neccessary action depending on what was hit
	 *
	 */
	public void checkCollisions() {
		Pacman p = localPlayer.getPacman();

		Iterable<Ghost> ghosts = aiPlayer.getActorList();

		for (Ghost g : ghosts) {
			if (p.collidesWith(g)) {
				// we hit a ghost - die if the ghost is active
				if (g.isActive()) {
					gui.die.playAndWait(5);
					restart();
				} else { // else send it to the corral
					gui.eatingGhosts.play();
					g.sendToCorral();
				}
			}
		}
		// eat the toekn
		for (Token t : tokenList) { 
			if (p.collidesWith(t)) { 
				// is a super token-> make the ghosts vulnerable
				if (t.isSuperToken()) {
					for (Ghost g : ghosts) {
						g.makeVulnerable();
					}
					vulnerableGhostTimer = new Timer(true);
					vulnerableGhostTimer.schedule(new RecoveringTimerTask(), 7000);
				}
				t.eat();
			}
		}
	}
	/**
	 * TimerTask that is fired when all the ghosts should be put into Recovering 
	 * mode. Also schedules the timer to make the ghosts active again
	 * @author Alec Berntson
	 *
	 */
	class RecoveringTimerTask extends TimerTask {
		public void run() {
			for (Ghost g : aiPlayer.getActorList()) {
				g.makeRecovering();
			}
			vulnerableGhostTimer.schedule(new ActiveTimerTask(), 3000);
		}
	}

	/**
	 * TimerTake that is fired when all the ghosts should be put into Active mode
	 * @author Alec Berntson
	 *
	 */
	class ActiveTimerTask extends TimerTask {
		public void run() {
			for (Ghost g : aiPlayer.getActorList()) {
				g.makeActive();
			}
		}
	}

	/**
	 * Restarts the game back to the way it was at the start
	 *
	 */
	private void restart() {
		gui.stopAnimation();
		for (Actor a : actorList) a.reset();
		gui.repaint();
		new Timer().schedule(new TimerTask() {
			public void run() {
				gui.intermission.playAndWait(10);
				gui.restartAnimation();
			}
		}, 1000);
	}

	/**
	 * Check if we have won
	 */
	public boolean isWinner() {
		return false;
	}

	/**
	 * @return the Phase that this game is in
	 */
	public Phase getPhase() {
		return phase;
	}
	/**
	 * return the Map used by this game
	 * @return
	 */
	public PacmanMap getMap(){
		return map;
	}

}
