package a4;

import java.net.*;

/**
 * Messages sent to a remote host should be encapsulated with
 * {@code Message} objects.
 * 
 * Do not create {@code Message} objects directly. Instead, use one of the
 * static methods provided; each corresponds to a valid protocol command.
 */
public final class Message {

   private final Command command;
   private final String[] args;

   private Message(Command command, String... args) {
      if (command == null || args == null) {
         throw new NullPointerException();
      }
      for (int i = 0; i < args.length; i++) {
         if (args[i] == null) {
            throw new NullPointerException();
         }
      }
      this.command = command;
      this.args = args;
   }

   /**
    * Returns the command in this message.
    * @return the command
    */
   public Command getCommand() {
      return command;
   }

   public String[] getArgs() {
      return args;
   }

   /**
    * Returns a {@code String} representation of this message suitable for
    * sending over the network.
    */
   public String toString() {
      StringBuilder s = new StringBuilder(command.toString());
      for (Object arg : args) {
         s.append("\t");
         s.append(encode(arg.toString()));
      }
      return s.toString();
   }

   private static String encode(String s) {
      try {
         return URLEncoder.encode(s, "UTF-8");
      } catch (Exception e) {
         return s;
      }
   }

   private static String decode(String s) {
      try {
         return URLDecoder.decode(s, "UTF-8");
      } catch (Exception e) {
         return s;
      }
   }

   /**
    * Parses a {@code String} into a message.
    */
   public static Message parse(String s) {
      String[] data = s.split("[\\t]+");
      if (data.length < 1) {
         throw new IllegalArgumentException();
      }
      Command type = Command.valueOf(data[0].toUpperCase());
      String[] args = new String[data.length - 1];
      for (int i = 1; i < data.length; i++) {
         args[i - 1] = decode(data[i]);
      }
      return new Message(type, args);
   }
   
   /*
    * send messages on a connection
    */
   public static void refresh(Connection remoteConnection) {
      remoteConnection.send(new Message(Command.REFRESH));
   }
   
   public static void refresh(Iterable<Connection> remoteConnections) {
      for (Connection remoteConnection : remoteConnections) {
         refresh(remoteConnection);
      }
   }
   
   public static void requestDownload(Connection remoteConnection, String file) {
      remoteConnection.send(new Message(Command.DOWNLOAD, file));
   }

   public static void sendFileList(Connection remoteConnection, String[] list) {
      remoteConnection.send(new Message(Command.FILELIST, list));
   }

   public static void sendText(Connection remoteConnection, String text) {
      remoteConnection.send(new Message(Command.TEXT, text));
   }

   public static void sendError(Connection remoteConnection, String message) {
      remoteConnection.send(new Message(Command.ERROR, message));
   }

   public static void sendDownloadReady(Connection remoteConnection, String file) {
      remoteConnection.send(new Message(Command.READY, file));
   }

   public static void sendDone(Connection remoteConnection) {
      remoteConnection.send(new Message(Command.DONE));
   }

}
