package a4;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.image.BufferedImage;
import java.util.Enumeration;
import java.util.Vector;

import javax.imageio.ImageIO;
import javax.swing.AbstractAction;
import javax.swing.AbstractButton;
import javax.swing.ButtonGroup;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JColorChooser;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;
import javax.swing.UIManager;

/*
 * Graphical user interface.
 */
public class GUI {

   final JFrame frame = new JFrame(); // main frame
   private Vector<String> listData = new Vector<String>();
   private final JList fileList = new JList(listData); // for displaying files
   private final JLabel messageArea; // for displaying messages
   private final HelpDialog helpDialog = new HelpDialog();
   private final Host host;
   /*
    * DEFAULT_IP is the IP address that appears in the connect box.
    * If you are testing on 2 computers, designate one of them
    * as the server and set DEFAULT_IP to be that computer's address
    * (use ipconfig from the commandline to get it).
    * 127.1.1.0 is the local host address, so you can create a
    * client and server on the same computer to test. 
    */
   public static final String DEFAULT_IP = "127.1.1.0";

   // these actions are shared by menu and button objects
   
   /*
    * server only
    */
   private final AbstractAction startAction = new AbstractAction("Start") {
      public void actionPerformed(ActionEvent e) {
         ((Server)host).start();
      }
   };
   
   /*
    * client only
    */
   private final AbstractAction connectAction = new AbstractAction("Connect...") {
      public void actionPerformed(ActionEvent e) {
         ((Client)host).connect();
      }
   };
   private final AbstractAction downloadAction = new AbstractAction("Download") {
      public void actionPerformed(ActionEvent e) {
         ((Client)host).download();
      }
   };
   
   /*
    * both server and client
    */
   private final AbstractAction refreshAction = new AbstractAction("Refresh") {
      public void actionPerformed(ActionEvent e) {
         host.refresh();
      }
   };

   // initialize user interface
   public GUI(Host host, Point location) {
      this.host = host;
      try {
         UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
      } catch (Exception exc) {}
      frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
      frame.setTitle(isClient()? "Client":"Server");

      // create menu bar
      JMenuBar menuBar = new JMenuBar();
      // file menu
      JMenu fileMenu = new JMenu("Action");
      if (isClient()) {
         fileMenu.add(new JMenuItem(connectAction));
         fileMenu.add(new JMenuItem(refreshAction));
         fileMenu.add(new JMenuItem(downloadAction));
      } else {
         fileMenu.add(new JMenuItem(startAction));
         fileMenu.add(new JMenuItem(refreshAction));
      }
      fileMenu.addSeparator();
      JMenuItem byeMenuItem = new JMenuItem("Quit");
      byeMenuItem.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent e) {
            frame.dispose();
         }
      });
      fileMenu.add(byeMenuItem);
      menuBar.add(fileMenu);

      // options menu
      JMenu optionsMenu = new JMenu("Options");

      JMenu colorMenu = new JMenu("Color");
      ButtonGroup colorGroup = new ButtonGroup();
      ColorRadioButton c = new ColorRadioButton("White/Black", Color.white, Color.black);
      setColors(c.background, c.foreground);
      c.setSelected(true);
      colorGroup.add(c);
      colorGroup.add(new ColorRadioButton("Ithaca Sunset",
            new Color(204, 204, 204), new Color(0, 0, 0)));
      colorGroup.add(new ColorRadioButton("Spruce Forest",
            new Color(0, 45, 45), new Color(32, 196, 112)));
      colorGroup.add(new ColorRadioButton("Ocean Breeze",
            new Color(72, 200, 200), new Color(0, 20, 86)));
      colorGroup.add(new ColorRadioButton("681 Web Site",
            new Color(0x33, 0x33, 0x55), new Color(0xFF, 0xFF, 0x99)));
      colorGroup.add(new ColorRadioButton("682 Web Site",
            new Color(0xCC, 0xCC, 0x99), new Color(0x99, 0x00, 0x00)));
      colorGroup.add(new ColorRadioButton("Cornell",
            new Color(255, 255, 255), new Color(179, 27, 27)));
      colorGroup.add(new ColorRadioButton("Custom...", Color.lightGray, Color.lightGray));
      Enumeration<AbstractButton> e = colorGroup.getElements();
      while (e.hasMoreElements()) colorMenu.add(e.nextElement());
      optionsMenu.add(colorMenu);

      menuBar.add(optionsMenu);

      // help menu
      JMenu helpMenu = new JMenu("Help");
      JMenuItem item = new JMenuItem("Get help...");
      item.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent evt) {
            helpDialog.display();
         }
      });
      helpMenu.add(item);

      helpMenu.addSeparator();

      item = new JMenuItem("About...", KeyEvent.VK_A);
      item.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent evt) {
            final String[] aboutMsg = {
                  "211ster Version 1.0",
                  "Copyright \u00a9 2007 Cornell University" };
            JOptionPane.showMessageDialog(frame, aboutMsg, "About 211ster",
                  JOptionPane.INFORMATION_MESSAGE);
         }
      });
      helpMenu.add(item);

      menuBar.add(helpMenu);
      frame.setJMenuBar(menuBar);

      // create toolbar
      JPanel toolBar = new JPanel(new FlowLayout(FlowLayout.LEFT));
      toolBar.setBackground(Color.lightGray);
      if (isClient()) {
         toolBar.add(new JButton(connectAction));
         toolBar.add(new JButton(refreshAction));
         toolBar.add(new JButton(downloadAction));
      } else {
         toolBar.add(new JButton(startAction));
         toolBar.add(new JButton(refreshAction));
      }
      frame.add(toolBar, BorderLayout.NORTH);

      // initialize file list
      fileList.setFont(Font.decode("Courier"));
      fileList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
      JScrollPane scrollPane = new JScrollPane(fileList);
      scrollPane.setPreferredSize(new Dimension(500, 100));
      frame.add(scrollPane, BorderLayout.CENTER);
      clearFiles();

      // create message area
      messageArea = new JLabel("", JLabel.LEFT);
      messageArea.setBackground(Color.darkGray);
      frame.add(messageArea, BorderLayout.SOUTH);
      clearMessage();

      // display the window
      setDisconnectedState();
      frame.pack();
      frame.setLocation(location);
      frame.setVisible(true);
   }

   void setConnectedState() {
      startAction.setEnabled(false);
      connectAction.setEnabled(false);
      refreshAction.setEnabled(true);
      downloadAction.setEnabled(true);
   }

   void setDisconnectedState() {
      startAction.setEnabled(true);
      connectAction.setEnabled(true);
      refreshAction.setEnabled(false);
      downloadAction.setEnabled(false);
   }

   private void setColors(Color b, Color f) {
      fileList.setBackground(b);
      fileList.setForeground(f);
   }

   void appendFile(String s) {
      synchronized (listData) {
         listData.add(s);
         fileList.setListData(listData);
      }
   }

   void clearFiles() {
      synchronized (listData) {
         listData.clear();
         fileList.setListData(listData);
      }
   }

   void appendFiles(String[] v) {
      synchronized (listData) {
         for (String s : v) {
            listData.add(s);
         }
         fileList.setListData(listData);
      }
   }

   void displayFiles(String[] v) {
      clearFiles();
      appendFiles(v);
   }

   String[] getListData() {
      final String[] x = new String[0];
      return listData.toArray(x);
   }

   String getSelectedFile() {
      return (String)fileList.getSelectedValue();
   }

   public void message(String s) {
      messageArea.setText(" " + s);
   }

   private void clearMessage() {
      message("");
   }
   
   private boolean isClient() {
      return host instanceof Client;
   }

   String getServerIP() {
      return (String)JOptionPane.showInputDialog(frame,
            "Enter the IP address of the server", "Enter Server IP",
            JOptionPane.PLAIN_MESSAGE, null, null, DEFAULT_IP);
   }
   
   void error(String title, String message) {
      JOptionPane.showMessageDialog(frame, message, title, JOptionPane.ERROR_MESSAGE);
   }

   private class ColorRadioButton extends JRadioButtonMenuItem implements ActionListener {
      Color background;
      Color foreground;

      ColorRadioButton(String s, Color b, Color f) {
         super(s);
         background = b;
         foreground = f;
         setActionCommand(s);
         addActionListener(this);
         if (!s.equals("Custom...")) setIcon(createIcon(b, f));
      }

      Icon createIcon(Color background, Color foreground) {
         Image image = new BufferedImage(12, 12, BufferedImage.TYPE_INT_RGB);
         Graphics graphics = image.getGraphics();
         graphics.setColor(background);
         graphics.fillRect(0, 0, 12, 12);
         graphics.setColor(foreground);
         graphics.fillRect(5, 0, 2, 12);
         graphics.fillRect(0, 5, 12, 2);
         return new ImageIcon(image);
      }

      public void actionPerformed(ActionEvent e) {
         if (e.getActionCommand().equals("Custom...")) {
            Color bc = JColorChooser.showDialog(frame, "Choose Background Color", background);
            if (bc == null) return;
            Color fc = JColorChooser.showDialog(frame, "Choose Foreground Color", foreground);
            if (fc == null) return;
            background = bc;
            foreground = fc;
            setIcon(createIcon(bc, fc));
         }
         setColors(background, foreground);
      }
   }

   private class HelpDialog extends JDialog {
      BufferedImage logo = null;
      int width = 200, height = 200;

      HelpDialog() {
         super(frame, "Help!");
         try {
            logo = ImageIO.read(ClassLoader.getSystemResourceAsStream("a4/munch.gif"));
            width = logo.getWidth();
            height = logo.getHeight();
         } catch (Exception e) {
            add(new JLabel("Help!", JLabel.CENTER));
         }
         setSize(width, height);
      }

      // display help dialog
      void display() {
         setLocation(frame.getX() + 100, frame.getY() + 50);
         setVisible(true);
      }

      public void paint(Graphics graphics) {
         graphics.setColor(getBackground());
         graphics.fillRect(0, 0, getSize().width, getSize().height);
         if (logo != null) {
            graphics.drawImage(logo, 0, 0, getBackground(), null);
         }
      }
   }

}

