package a4;

import java.io.*;
import java.net.*;
import java.util.*;
import javax.swing.*;

/**
 * A class for communications between two hosts. Sending messages is
 * done with the {@code send} method. To receive messages, you have two options:
 * implement a {@code MessageListener}, or use the {@code recv} method
 * directly. For a GUI application, it is recommended that you use the listener
 * approach.
 * 
 * When using with listeners, first create a new instance of {@code Connection}.
 * Then use {@code addMessageListener} to add your listener(s). Finally, call
 * {@code start} to initiate the message receiving process.
 * 
 * Each listener will be notified whenever a message is received via its
 * {@code messageReceived} method.
 * 
 * To stop listening for more messages, call {@code close}. (Note that this
 * closes the underlying connection, so you cannot use this instance of
 * {@code Connection} any more).
 * 
 * If you are not using listeners, do not call the {@code start} method. Simply
 * use {@code recv} when you expect a message. This will block until a message
 * is received.
 * 
 * NB: You should not use both listeners and the {@code recv} method directly.
 * If you do so, there is no guarantee as to which messages will be routed to
 * the listeners and which will be returned by direct {@code recv} calls.
 */
public class Connection extends Thread {

   private final Socket socket;
   private final PrintWriter out;
   private final BufferedReader in;
   private final List<MessageListener> listeners = new ArrayList<MessageListener>();

   private class MessageNotifier implements Runnable {

      private final MessageListener listener;
      private final Message message;

      public MessageNotifier(MessageListener listener, Message message) {
         this.listener = listener;
         this.message = message;
      }

      public void run() {
         listener.messageReceived(Connection.this, message);
      }
   }

   /**
    * Creates a new {@code Connection} with the specified socket.
    * 
    * @param socket
    *           the socket that is already connected to the remote host
    * @throws IOException
    *            if a socket error occurred
    */
   public Connection(Socket socket) throws IOException {
      this.socket = socket;
      out = new PrintWriter(new OutputStreamWriter(socket.getOutputStream(), "UTF-8"), true);
      in = new BufferedReader(new InputStreamReader(socket.getInputStream(), "UTF-8"));
   }

   /**
    * Creates a new {@code Connection} to the specified IP address and port
    * number.
    * 
    * @param address
    *           the IP address
    * @param port
    *           the port number
    * @throws IOException
    *            if a connection error occurred
    */
   public Connection(InetAddress address, int port) throws IOException {
      this(new Socket(address, port));
   }

   /**
    * Sends a message.
    * 
    * @param m
    *           the message to send
    * @throws IOException
    *            if a network error occurred
    */
   public void send(Message m) {
      if (m == null) {
         throw new NullPointerException();
      }
      synchronized (out) {
         out.println(m);
      }
   }

   /**
    * Receives a message with no timeout. This method blocks until a message is
    * received.
    * 
    * @return the message received
    * @throws IOException
    *            if a network error occurred
    */
   public Message recv() throws IOException {
      return recv(0);
   }

   /**
    * Receives a message with a timeout. If the timeout occurs before a message
    * has been received, it returns {@code null}.
    * 
    * @param timeout
    *           timeout in milliseconds
    * @return the message received, or {@code null} if a timeout occurred
    * @throws IOException
    *            if a network error occurred
    */
   public Message recv(int timeout) throws IOException {
      String s = null;
      synchronized (in) {
         int oldTimeout = setTimeout(timeout);
         in.mark(1024); //mark input
         try {
            s = in.readLine();
         } catch (SocketTimeoutException e) {
            in.reset(); //reset to mark
            return null;
         } finally {
            setTimeout(oldTimeout);
         }
      }
      if (s == null) {
         throw new IOException();
      }

      try {
         return Message.parse(s);
      } catch (Exception e) {
         throw new IOException();
      }
   }

   /**
    * Closes the underlying connection and stops listening for messages.
    */
   public void close() {
      try {
         socket.close();
      } catch (IOException e) {}
   }

   /**
    * Adds a listener for receiving messages.
    * 
    * @param l
    *           the listener to add
    */
   public void addMessageListener(MessageListener l) {
      synchronized (listeners) {
         listeners.add(l);
      }
   }

   /**
    * Runs the message receiving thread. Should not be called directly. Use
    * {@code start} to start the message receiver.
    */
   public void run() {
      try {
         while (!socket.isClosed()) {
            Message m = recv();
            synchronized (listeners) {
               for (MessageListener l : listeners) {
                  SwingUtilities.invokeLater(new MessageNotifier(l, m));
               }
            }
         }
      } catch (IOException e) {}
   }

   private int setTimeout(int timeout) {
      int oldTimeout = 0;
      try {
         oldTimeout = socket.getSoTimeout();
         socket.setSoTimeout(timeout);
      } catch (IOException e) {}
      return oldTimeout;
   }
   
   String getIP() {
      return socket.getInetAddress().toString();
   }
   
   int getPort() {
      return socket.getPort();
   }

   String getAddress() {
      return socket.getRemoteSocketAddress().toString();
   }

   String getLocalAddress() {
      try {
      return InetAddress.getLocalHost().toString();
      } catch (UnknownHostException uhe) {
         return socket.getLocalAddress().toString();
      }
   }

}
