package a4;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.net.InetAddress;
import java.util.Vector;

public class Client extends Host {
   
   Connection serverConnection;
   String clientID;
   File workingDirectory = new File(System.getProperty("user.dir"));

   Client(int port) throws IOException {
      this(port, 700, 200); //port for listening for p2p connections
   }

   Client(int port, int x, int y) throws IOException {
      super(port, x, y); //port for listening for p2p connections
   }

   /*
    * button and menu handlers
    */

   /*
    * Call this method to connect to the server.
    * This creates a new Connection object and starts
    * the message listener. The remote server must
    * already be running when this is called.
    */
   void connect() {
      String serverIP;
      while (true) {
         serverIP = gui.getServerIP();
         if (serverIP == null) return; //user clicked cancel
         gui.message("Connecting to " + serverIP);
         try {
            serverConnection = new Connection(InetAddress.getByName(serverIP), DEFAULT_PORT);
            serverConnection.addMessageListener(this);
            serverConnection.start();
            break;
         } catch (Exception e) {
            gui.error("Connection Error", "Error Connecting: " + e.getMessage());
         }
      }
      gui.setConnectedState();
      clientID = serverConnection.getLocalAddress() + ":" + listenerPort;
      gui.message("Connected to " + serverConnection.getAddress());
      gui.frame.setTitle("Client " + clientID);
      super.start(); //start listening for p2p connections
   }
   
   Connection connectToPeer(InetAddress peerIP, int port) {
      gui.message("Connecting to " + peerIP);
      Connection peer = null;
      try {
         peer = new Connection(peerIP, port);
      } catch (IOException e) {
         gui.error("Connection Error", "Error Connecting: " + e.getMessage());
         if (peer != null) peer.close();
         return null;
      }
      gui.message("Connected to " + peer.getAddress());
      return peer;
   }

   void refresh() {
      gui.message("Requesting file list refresh from server");
      Message.refresh(serverConnection);
   }

   void download() {
      String file = gui.getSelectedFile();
      if (file == null) {
         gui.message("Select a file to download");
         return;
      }
      gui.message("Requesting download");
      //parse filename, ip, port
      RemoteFile rf = null;
      try {
         rf = new RemoteFile(file);
      } catch (IllegalArgumentException iae) {
         gui.error("Error", "Unrecognized filename or IP address");
         return;
      }
      Connection p2p = connectToPeer(rf.getAddress(), rf.getPort());
      String fileName = rf.getFileName();
      Message.requestDownload(p2p, fileName);
      Message m = null;
      try {
         m = p2p.recv(10000);
         if (m == null) throw new IOException("Transmission timeout");
         if (m != null && !m.getCommand().equals(Command.READY))
            throw new IOException("Transmission error, download aborted");
      } catch (IOException ioe) {
         gui.error("Error", ioe.getMessage());
         p2p.close();
         return;
      }
      gui.message("Receiving " + fileName + " from " + p2p.getIP());
      File outFile = new File(workingDirectory, fileName);
      if (outFile.exists()) {
         //find a suitable output file name
         int j = fileName.lastIndexOf('.');
         String name = (j < 0)? fileName : fileName.substring(0, j);
         String tag = (j < 0)? "" : fileName.substring(j);
         for (j = 1; outFile.exists(); j++) {
            outFile = new File(workingDirectory, name + "(" + j + ")" + tag);
         }
      }
      
      FileWriter dataOut = null;
      try {
         dataOut = new FileWriter(outFile);
         while (true) {
            m = p2p.recv(10000);
            if (m == null) throw new IOException("Transmission timeout");
            if (m.getCommand().equals(Command.DONE)) break;
            if (!m.getCommand().equals(Command.TEXT)) throw new IOException("Transmission error, download aborted");
            String text = m.getArgs()[0];
            dataOut.write(text);
         } 
         dataOut.flush();
         dataOut.close();
         gui.message("Download successful");
      } catch (IOException ioe) {
         gui.error("Error", ioe.getMessage());
      }
      p2p.close();
   }
   
   public void close() {
      super.close();
      try {
         serverConnection.close();
      } catch (Exception e) {}
   }
   
   /*
    * message handlers
    */
   public void messageReceived(Connection connection, Message message) {
      switch (message.getCommand()) {
      case REFRESH:
         //collect local file list
         String[] fileList = workingDirectory.list(new FilenameFilter() {
            public boolean accept(File dir, String name) {
               return new File(dir, name).isFile();
            }
         });
         //append my ip address and listener port
         for (int i = 0; i < fileList.length; i++) {
            fileList[i] = clientID + " " + fileList[i];
         }
         Message.sendFileList(serverConnection, fileList);
         break;
      case FILELIST:
         //display list of available remote files
         Vector<String> v = new Vector<String>();
         for (String s : message.getArgs()) {
            RemoteFile rf = null;
            try {
               rf = new RemoteFile(s);
            } catch (IllegalArgumentException iae) {
               continue;
            }
            if (rf.getSocketAddress().equals(clientID)) continue;
            v.add(s);
         }
         gui.displayFiles(v.toArray(new String[0]));
         gui.message("File list refreshed");
         break;
      case DOWNLOAD:
         synchronized (this) {
            String fileName = message.getArgs()[0];
            gui.message("Sending " + fileName + " to " + connection.getIP());
            File file = new File(fileName);
            if (!file.exists()) {
               Message.sendError(connection, "File does not exist");
               break;
            }
            if (file.isDirectory()) {
               Message.sendError(connection, "File is a directory");
               break;
            }
            FileReader dataIn;
            try {
               dataIn = new FileReader(file);
            } catch (FileNotFoundException fnfe) {
               Message.sendError(connection, "File does not exist");
               break;
            }
            Message.sendDownloadReady(connection, fileName);
            char[] buf = new char[128];
            try {
               while (true) {
                  int bytesRead = dataIn.read(buf);
                  if (bytesRead == -1 || bytesRead == 0) break;
                  String msg = new String(buf, 0, bytesRead);
                  Message.sendText(connection, msg);
               }
               Message.sendDone(connection);
               dataIn.close();
               gui.message("File successfully sent");
            } catch (IOException e) {
               Message.sendError(connection, "Transmission error, download aborted");
            }
            connection.close();
         }
         break;
      case ERROR:
         gui.error("Error", (String)message.getArgs()[0]);
         break;
      default:
         gui.error("Error", "Unrecognized message");
      }
   }
   
}
