package a3;
/*
 IO.java
 This file contains low-level file I/O routines.
 */

import java.io.*;

class IO {
   private static BufferedInputStream inStream;
   private static BufferedOutputStream outStream;
   static int bytesRead;

   // create an input stream from a given file
   static void openForRead(File f) throws IOException {
      inStream = new BufferedInputStream(new FileInputStream(f));
      bytesRead = 0;
   }

   // create an output stream to a given file
   static void openForWrite(File f) throws IOException {
      outStream = new BufferedOutputStream(new FileOutputStream(f));
   }

   // close current input stream
   static void closeForRead() throws IOException {
      inStream.close();
   }

   // flush output byte buffer
   static void flush() throws IOException {
      outStream.flush();
   }

   // close current output stream
   static void closeForWrite() throws IOException {
      flush(); // flush buffer first
      outStream.close();
   }

   // close current input and output streams
   static void cleanup() throws IOException {
      closeForRead();
      closeForWrite();
   }

   // read one byte from current input stream
   static int get() throws IOException {
      int b = -1;
      b = inStream.read();
      if (b != -1) bytesRead++;
      return b;
   }

   // write one byte to current output stream
   static void put(int k) throws IOException {
      outStream.write(k);
   }

   // get a ByteString of length n from current input stream
   static ByteString getByteString(int n) throws IOException {
      byte[] b = new byte[n];
      if (inStream.read(b) < n) throw new IOException("Unexpected end of file");
      bytesRead += n;
      return new ByteString(b);
   }

   // write a ByteString to current output stream
   // can be used for compressed data
   static void putByteString(ByteString s) throws IOException {
      for (int i = 0; i < s.length(); i++) {
         put((int)s.byteAt(i));
      }
   }

   // get an int (4 bytes) from current input stream
   // should only be used for header info, never for compressed data
   static int getInteger() throws IOException {
      int k = 0;
      for (int i = 0; i < 4; i++) {
         int j = get();
         if (j < 0)
            throw new IOException("Unexpected end of file");
         k = k << 8 | j;
      }
      return k;
   }

   // write an int (4 bytes) to current output stream
   // should only be used for header info, never for compressed data
   static void putInteger(int n) throws IOException {
      put(n >> 24 & 255);
      put(n >> 16 & 255);
      put(n >> 8 & 255);
      put(n & 255);
   }
   
   static final long MASK = (1L << 32) - 1; //mask for low order word

   // get a long (8 bytes) from current input stream
   // should only be used for header info, never for compressed data
   static long getLong() throws IOException {
      return Long.rotateLeft(getInteger() & MASK, 32) | (getInteger() & MASK);
   }

   // write a long (8 bytes) to current output stream
   // should only be used for header info, never for compressed data
   static void putLong(long n) throws IOException {
      putInteger((int)(Long.rotateLeft(n,32) & MASK));
      putInteger((int)(n & MASK));
   }

   // copy from input stream to output stream until end of file
   // return number of bytes copied
   static int copy() throws IOException {
      int count = 0;
      int x = inStream.read();
      while (x != -1) {
         count++;
         outStream.write(x);
         x = inStream.read();
      }
      return count;
   }

   // copy from input stream to output stream
   // up to m bytes or end of file
   // return number of bytes copied
   static int copy(int m) throws IOException {
      int count;
      for (count = 0; count < m; count++) {
         int x = inStream.read();
         if (x == -1)
            break;
         outStream.write(x);
      }
      return count;
   }
}
