package a3;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.FileDialog;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.Enumeration;

import javax.imageio.ImageIO;
import javax.swing.AbstractAction;
import javax.swing.AbstractButton;
import javax.swing.ButtonGroup;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JColorChooser;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JTextArea;
import javax.swing.UIManager;

/*
 * Graphical user interface for the 211 archive program.
 */
class GUI {

   static GUI gui;
   private final JFrame frame = new JFrame("211Zip"); // main frame
   private final JTextArea fileList = new JTextArea(); // for displaying files in the archive
   private final JLabel messageArea; // for displaying messages
   private final MyHelpDialog helpDialog = new MyHelpDialog();

   // these actions are shared by menu and button objects
   private final AbstractAction newAction = new AbstractAction("New Archive...") {
      public void actionPerformed(ActionEvent e) {
         newArchive();
      }
   };
   private final AbstractAction openAction = new AbstractAction(
         "Open Archive...") {
      public void actionPerformed(ActionEvent e) {
         openArchive();
      }
   };
   private final AbstractAction closeAction = new AbstractAction(
         "Close Archive") {
      public void actionPerformed(ActionEvent e) {
         closeArchive();
      }
   };
   private final AbstractAction addAction = new AbstractAction("Add File...") {
      public void actionPerformed(ActionEvent e) {
         addFile();
      }
   };
   private final AbstractAction extractAction = new AbstractAction(
         "Extract Files...") {
      public void actionPerformed(ActionEvent e) {
         extractFiles();
      }
   };

   // initialize user interface
   GUI() {
      gui = this;
      try {
         UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
      } catch (Exception e) {}
      frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

      // create menu bar
      JMenuBar menuBar = new JMenuBar();
      // file menu
      JMenu fileMenu = new JMenu("File");
      fileMenu.add(new JMenuItem(newAction));
      fileMenu.add(new JMenuItem(openAction));
      fileMenu.add(new JMenuItem(closeAction));
      fileMenu.addSeparator();
      fileMenu.add(new JMenuItem(addAction));
      fileMenu.add(new JMenuItem(extractAction));
      fileMenu.addSeparator();
      JMenuItem byeMenuItem = new JMenuItem("Quit");
      byeMenuItem.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent e) {
            System.exit(0);
         }
      });
      fileMenu.add(byeMenuItem);
      menuBar.add(fileMenu);

      // options menu
      JMenu optionsMenu = new JMenu("Options");

      JMenu colorMenu = new JMenu("Color");
      ButtonGroup colorGroup = new ButtonGroup();
      ColorRadioButton c = new ColorRadioButton("White/Black", Color.white,
            Color.black);
      setColors(c.background, c.foreground);
      c.setSelected(true);
      colorGroup.add(c);
      colorGroup.add(new ColorRadioButton("Ithaca Sunset",
            new Color(204, 204, 204), new Color(0, 0, 0)));
      colorGroup.add(new ColorRadioButton("Spruce Forest",
            new Color(0, 45, 45), new Color(32, 196, 112)));
      colorGroup.add(new ColorRadioButton("Ocean Breeze",
            new Color(72, 200, 200), new Color(0, 20, 86)));
      colorGroup.add(new ColorRadioButton("681 Web Site",
            new Color(0x33, 0x33, 0x55), new Color(0xFF, 0xFF, 0x99)));
      colorGroup.add(new ColorRadioButton("682 Web Site",
            new Color(0xCC, 0xCC, 0x99), new Color(0x99, 0x00, 0x00)));
      colorGroup.add(new ColorRadioButton("Cornell",
            new Color(255, 255, 255), new Color(179, 27, 27)));
      colorGroup.add(new ColorRadioButton("Custom...", Color.lightGray, Color.lightGray));
      Enumeration<AbstractButton> e = colorGroup.getElements();
      while (e.hasMoreElements()) colorMenu.add(e.nextElement());
      optionsMenu.add(colorMenu);

      menuBar.add(optionsMenu);

      // help menu
      JMenu helpMenu = new JMenu("Help");
      JMenuItem item = new JMenuItem("Get help...");
      item.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent e) {
            helpDialog.display();
         }
      });
      helpMenu.add(item);

      helpMenu.addSeparator();

      item = new JMenuItem("About...", KeyEvent.VK_A);
      item.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent e) {
            final String[] aboutMsg = { "211Zip Version 1.0",
                  "Copyright \u00a9 2007 Cornell University" };
            JOptionPane.showMessageDialog(frame, aboutMsg, "About 211Zip",
                  JOptionPane.INFORMATION_MESSAGE);
         }
      });
      helpMenu.add(item);

      menuBar.add(helpMenu);
      frame.setJMenuBar(menuBar);

      // create toolbar
      JPanel toolBar = new JPanel(new FlowLayout(FlowLayout.LEFT));
      toolBar.setBackground(Color.lightGray);
      toolBar.add(new JButton(newAction));
      toolBar.add(new JButton(openAction));
      toolBar.add(new JButton(closeAction));
      toolBar.add(new JButton(addAction));
      toolBar.add(new JButton(extractAction));
      frame.add(toolBar, BorderLayout.NORTH);

      // init text area for file list
      fileList.setEditable(false);
      fileList.setFont(Font.decode("Courier"));
      fileList.setPreferredSize(new Dimension(0, 100));
      frame.add(fileList, BorderLayout.CENTER);
      clearFiles();

      // create message area
      messageArea = new JLabel("", JLabel.LEFT);
      messageArea.setBackground(Color.darkGray);
      frame.add(messageArea, BorderLayout.SOUTH);
      clearMessage();

      // display the window
      setClosedState();
      frame.pack();
      frame.setLocation(200, 200);
      frame.setVisible(true);
   }

   // set menus and buttons for when there is an archive open
   private void setOpenState() {
      closeAction.setEnabled(true);
      addAction.setEnabled(true);
      extractAction.setEnabled(true);
   }

   // set menus and buttons for when there is no archive open
   private void setClosedState() {
      closeAction.setEnabled(false);
      addAction.setEnabled(false);
      extractAction.setEnabled(false);
   }

   private void setColors(Color b, Color f) {
      fileList.setBackground(b);
      fileList.setForeground(f);
   }

   private void setTitle(String s) {
      frame.setTitle(s);
   }

   private void listFiles(String s) {
      if (s.equals("")) s = "No files";
      fileList.setText(s);
   }

   private void clearFiles() {
      fileList.setText("");
   }

   private void message(String s) {
      messageArea.setText(" " + s);
   }

   private void clearMessage() {
      message("");
   }

   /*
    * Various GUI objects and event handlers
    */

   // create new archive
   private void newArchive() {
      FileDialog openDialog = new FileDialog(frame, "New Archive", FileDialog.SAVE);
      openDialog.setDirectory(System.getProperty("user.dir"));
      openDialog.setVisible(true);
      if (openDialog.getFile() == null)
         return; // user clicked cancel
      Archive archive = new Archive(openDialog.getDirectory() + openDialog.getFile());
      archive.nFiles = 0; // no files
      // write out header info
      try {
         archive.writeHeader(archive);
         IO.closeForWrite();
      } catch (IOException ioe) {
         message("Cannot write archive: " + ioe.getMessage());
         return;
      }
      try {
         archive.read(); // read it back in
         Archive.archive = archive;
      } catch (IOException ioe) {
         message("Cannot read archive: " + ioe.getMessage());
      }
   }

   // called from archive after it is read in to set gui state
   void initArchive(String name, String files) {
      listFiles(files); // display file info
      setTitle(name);
      clearMessage();
      setOpenState();
   }

   // open existing archive
   private void openArchive() {
      FileDialog openDialog = new FileDialog(frame, "Open Archive", FileDialog.LOAD);
      openDialog.setDirectory(System.getProperty("user.dir"));
      openDialog.setVisible(true);
      if (openDialog.getFile() == null) return; // user clicked cancel
      Archive archive = new Archive(openDialog.getDirectory() + openDialog.getFile());
      try {
         archive.open();
         Archive.archive = archive;
      } catch (Exception ioe) {
         message(ioe.getMessage());
      }
   }

   // open existing archive
   private void closeArchive() {
      try {
         Archive.archive.close();
      } catch (IOException ioe) {
         message(ioe.getMessage());
      }
      setClosedState();
      clearFiles();
      clearMessage();
   }

   // open existing archive
   private void addFile() {
      FileDialog openDialog = new FileDialog(GUI.gui.frame, "Select File to Add", FileDialog.LOAD);
      openDialog.setVisible(true);
      if (openDialog.getFile() == null) return; // user clicked cancel
      File file = new File(openDialog.getDirectory() + openDialog.getFile());
      try {
         Archive.archive.addFile(file);
      } catch (IOException ioe) {
         message(ioe.getMessage());
      }
   }

   // extract files to temp directory
   private void extractFiles() {
      String dir = System.getProperty("java.io.tmpdir");
      message("Extracting files to " + dir + " ... ");
      try {
         Archive.archive.extract();
      } catch (IOException ioe) {
         message(ioe.getMessage());
         return;
      }
      message("Files extracted to " + dir);
   }

   private class ColorRadioButton extends JRadioButtonMenuItem implements ActionListener {
      Color background;
      Color foreground;

      ColorRadioButton(String s, Color b, Color f) {
         super(s);
         background = b;
         foreground = f;
         setActionCommand(s);
         addActionListener(this);
         if (!s.equals("Custom...")) setIcon(createIcon(b, f));
      }

      Icon createIcon(Color background, Color foreground) {
         Image image = new BufferedImage(12, 12, BufferedImage.TYPE_INT_RGB);
         Graphics graphics = image.getGraphics();
         graphics.setColor(background);
         graphics.fillRect(0, 0, 12, 12);
         graphics.setColor(foreground);
         graphics.fillRect(5, 0, 2, 12);
         graphics.fillRect(0, 5, 12, 2);
         return new ImageIcon(image);
      }

      public void actionPerformed(ActionEvent e) {
         if (e.getActionCommand().equals("Custom...")) {
            Color bc = JColorChooser.showDialog(frame, "Choose Background Color", background);
            if (bc == null) return;
            Color fc = JColorChooser.showDialog(frame, "Choose Foreground Color", foreground);
            if (fc == null) return;
            background = bc;
            foreground = fc;
            setIcon(createIcon(bc, fc));
         }
         setColors(background, foreground);
      }
   }

   private class MyHelpDialog extends JDialog {
      BufferedImage logo = null;
      int width = 200, height = 200;

      MyHelpDialog() {
         super(frame, "Help!");
         try {
            logo = ImageIO.read(ClassLoader.getSystemResourceAsStream("a3/munch.gif"));
            width = logo.getWidth();
            height = logo.getHeight();
         } catch (Exception e) {
            add(new JLabel("Help!", JLabel.CENTER));
         }
         setSize(width, height);
         setLocation(300, 300);
      }

      // display help dialog
      void display() {
         setVisible(true);
      }

      public void paint(Graphics graphics) {
         graphics.setColor(getBackground());
         graphics.fillRect(0, 0, getSize().width, getSize().height);
         if (logo != null) {
            graphics.drawImage(logo, 0, 0, getBackground(), null);
         }
      }
   }
}
