package a3;

/*
Archive.java
This file contains the class Archive and all its functionality.

The file format of an archive is as follows:

HEADER
8 bytes:  "<211Zip>"
4 bytes:  int n, number of files in archive
File 1 info:
  4 bytes:  int k1, length of name of file 1
  k1 bytes: string, name of file 1
  4 bytes:  int a1, address of compressed data for file 1
  4 bytes:  int u1, uncompressed size of file 1
File 2 info:
  4 bytes:  int k2, length of name of file 2
  k2 bytes: string, name of file 2
  4 bytes:  int a2, address of compressed data for file 2
  4 bytes:  int u2, uncompressed size of file 2
...
File n info:
  4 bytes:  int kn, length of name of file n
  kn bytes: string, name of file n
  4 bytes:  int an, address of compressed data for file n
  4 bytes:  int un, uncompressed size of file n

DATA
a1 to a2-1: compressed data for file 1
a2 to a3-1: compressed data for file 2
...
an to end of file: compressed data for file n
*/

import java.io.*;
import java.util.Date;
import java.util.Formatter;

class Archive extends File {
   private static final int MAXNFILES = 256; //max # of files
   private static final int MAXFILENAMELENGTH = 256;
   private static final long MAGIC = new ByteString("<211Zip>").toLong();
   int nFiles; //number of files in this archive
   private String[] fileName = new String[MAXNFILES]; //names of files
   private int[] address = new int[MAXNFILES]; //starting byte in archive
   private int[] uncompressedSize = new int[MAXNFILES]; //uncompressed size of file
   private long[] lastModified = new long[MAXNFILES]; //date last modified
   private int headerLength; //archive header length
   static Archive archive = null; //the current archive
       
   Archive(String fileName) {
      super(fileName);
   }
   
   //open this archive
   void open() throws IOException {
      if (!exists()) { //archive must exist
         throw new FileNotFoundException("Archive does not exist");
      }
      else if (!isFile() || length() == 0) { //must be a nonempty file
         throw new IllegalArgumentException("Not a valid archive");
      }
      else if (!canRead()) { //must be readable
         throw new IOException("Archive cannot be read");
      }
      else if (!canWrite()) { //and writable
         throw new IOException("Archive cannot be written");
      }
      read(); //read it in
   }
   
   //read in archive
   void read() throws IOException {
      int i,m;
      IO.openForRead(this);
      if (IO.getLong() != MAGIC) { //1st 8 bytes
         throw new IllegalArgumentException("Not a valid archive");
      }
      nFiles = IO.getInteger(); //number of files in archive
      if (nFiles < 0 || nFiles > MAXNFILES) {
         throw new IllegalArgumentException("Not a valid archive");
      }
      for (i = 0; i < nFiles; i++) { //for each file in archive
         m = IO.getInteger(); //length of file name
         if (m < 0 || m > MAXFILENAMELENGTH) {
            throw new IllegalArgumentException("Not a valid archive");
         }
         fileName[i] = IO.getByteString(m).toString(); //get file name
         address[i] = IO.getInteger(); //byte address in compressed file
         uncompressedSize[i] = IO.getInteger();
         lastModified[i] = IO.getLong();
      }
      headerLength = IO.bytesRead; //length of header
      
      //assemble file info for display
      int max = 0;
      for (i = 0; i < nFiles; i++) {
         if (max < fileName[i].length()) max = fileName[i].length();
      }
      String formatString = "%-" + max + "s %d/%d %2.1f%% %s\n";
      StringBuilder sb = new StringBuilder();
      Formatter formatter = new Formatter(sb);
      for (i = 0; i < nFiles; i++) {
         double compressionRatio = Math.round(1000.0 - compressedSize(i)*1000.0/uncompressedSize[i])/10.0;
         formatter.format(formatString,
               fileName[i],
               compressedSize(i),
               uncompressedSize[i],
               compressionRatio,
               new Date(lastModified[i]).toString());
      }
      GUI.gui.initArchive(getAbsolutePath(), sb.toString());
   }
   
   //add a file to the archive
   void addFile(File f) throws IOException {
      if (!f.exists()) { //file better exist
         throw new FileNotFoundException("File does not exist");
      }
      if (!f.isFile()) { //and better not be a directory
         throw new IllegalArgumentException("Not a valid file");
      }
      if (!f.canRead()) { //and better be readable
         throw new IOException("File cannot be read");
      }
      if (nFiles >= MAXNFILES) {
         throw new IOException("Archive is full");
      }
      File tempArchive = File.createTempFile("zip", null); //temp archive
      File fCompressed = File.createTempFile("com", null); //compressed file
      fileName[nFiles] = f.getName(); //add new file info to header
      address[nFiles] = (int)length() - headerLength;
      uncompressedSize[nFiles] = (int)f.length();
      lastModified[nFiles++] = f.lastModified();
      Zip.codec.compress(f, fCompressed); //compress the new file
      writeHeader(tempArchive); //write new archive header to temp file
      IO.copy(); //copy previous compressed files to temp file
      IO.closeForRead();
      IO.openForRead(fCompressed); //add new compressed file
      IO.copy();
      IO.cleanup();
      fCompressed.delete();
      IO.openForRead(tempArchive); //now copy back to archive
      IO.openForWrite(this);
      IO.copy(); //should really do this with a system call
      IO.cleanup();
      tempArchive.delete(); //delete temp file
      read(); //reread the archive
   }

   //extract files from the archive to temp directory
   void extract() throws IOException {
      File dir = new File(System.getProperty("java.io.tmpdir"));
      String dirName = dir.getAbsolutePath() + File.separator;
      if (!dir.exists()) { //check that directory exists and is writable
         throw new FileNotFoundException("Directory " + dirName + " does not exist");
      }
      else if (dir.isFile()) {
         throw new IllegalArgumentException(dirName + " is not a directory");
      }
      else if (!dir.canRead()) {
         throw new IOException("Cannot read " + dirName);
      }
      else if (!dir.canWrite()) {
         throw new IOException("Cannot write to " + dirName);
      }
      File[] fCompressed = new File[nFiles];
      for (int i = 0; i < nFiles; i++) { //for each file in archive
         fCompressed[i] = File.createTempFile(fileName[i] + "-com", null);
         IO.openForWrite(fCompressed[i]);
         IO.copy(compressedSize(i)); //copy to temp compressed file
         IO.closeForWrite();
      }
      IO.closeForRead();
      for (int i = 0; i < nFiles; i++) { //decompress
         File f = new File(dirName + fileName[i]);
         Zip.codec.decompress(fCompressed[i], f);
         fCompressed[i].delete(); //clean up
         f.setLastModified(lastModified[i]);
      }
      read(); //reread the archive
   }
   
   //get compressed size of i-th file in archive
   //address[i+1] - address[i] if not the last file
   //end of file - address[i] if the last file   
   private int compressedSize(int i) {
      return ((i < nFiles-1)? address[i+1] : (int)length()-headerLength) - address[i];
   }
   
   //write out header to archive
   void writeHeader(File archive) throws IOException {
      IO.openForWrite(archive);
      IO.putLong(MAGIC);
      IO.putInteger(nFiles); //number of files
      for (int i = 0; i < nFiles; i++) {
         IO.putInteger(fileName[i].length()); //length of filename string
         IO.putByteString(new ByteString(fileName[i])); //filename
         IO.putInteger(address[i]); //address of compressed data for this file
         IO.putInteger(uncompressedSize[i]);
         IO.putLong(lastModified[i]);
      }
      IO.flush();
   }
   
   //close the archive   
   void close() throws IOException {
      archive = null;
      IO.closeForRead();
   }
}
