import java.util.*;
import java.io.*;

/**
 This class is used to classify the part of speech of
 input words and to generate random words of
 given category using the dictionary specified
 in the file "dictionary.txt"
 @see TokenCategory
*/
class Dictionary {
	private Random prng = new Random();

	/**
	 This method constructs the dictionary but reading in the
	 word classifications from the disk
	 @throws DictionaryException if the dictionary file was not
    found or somehow malformed
	*/
	Dictionary() throws DictionaryException {
	
		//Read in from the 'dictionary file'
		try {
			BufferedReader in = new BufferedReader(new FileReader("dictionary.txt"));
			String line;
			while ((line = in.readLine()) != null) {
				line = line.trim();
				
				//Skip comments, empty lines
				if (line.startsWith("#")) continue;
				if (line.length() == 0) continue;
				
				//Split in two on space..
				int pos = line.indexOf(' ');
				if (pos == -1 || pos == line.length() - 1)
					throw new DictionaryException("Invalid line " + 
						line + " in dictionary");
						
            String word = line.substring(0, pos).trim();
            String cat  = line.substring(pos + 1).trim();
            //First letter is upper case
            cat = cat.substring(0,1).toUpperCase() + cat.substring(1).toLowerCase();
				TokenCategory tcat = null;
            try {
               tcat = TokenCategory.valueOf(cat);
            } catch (IllegalArgumentException e) {
               // Seems like something invalid was specified..
               throw new DictionaryException("Unknown lexical category " +
                  cat + " for word " + word);
            }
            
            // End of file and unknown should not be set via
            // the dictionary file (but aliases for blank are fine..)
            if (tcat == TokenCategory.Unknown ||
                tcat == TokenCategory.EndOfFile)
               throw new DictionaryException("Invalid attempt to " +
                  "customize lexical category " + tcat);

            addToDictionary(word, tcat);
			}
		}
		catch (IOException e) {
			throw new DictionaryException(
				"Error reading in dictionary: " + e.getMessage());
		}
      
      final EnumSet<TokenCategory> builtins = EnumSet.of(
            TokenCategory.Blank,
            TokenCategory.Unknown,
            TokenCategory.Error,
            TokenCategory.EndOfFile);
		
		//Sanity check -- there must be at least one word of each category,
		//except for the 4 builtins
		for (TokenCategory cat : EnumSet.complementOf(builtins)) {
			if (!words.containsKey(cat))
				throw new DictionaryException("The dictionary does not "
					+ "contain words of category:" + cat);
		}
	}

	//This map is used to store all words of the given category,
	//to make it easier to randomly generate one
	private Map<TokenCategory, ArrayList<String>> words =
		new HashMap<TokenCategory, ArrayList<String>>();

	//This map is used to classify words
	private Map<String, TokenCategory> wordCategories =
		new HashMap<String, TokenCategory>();

	//Helper method to update data structures for both
	//looking up categories of words and generatign words of categories
	private void addToDictionary(String word, TokenCategory tcat) {
		String wordKey = word.toLowerCase();
		ArrayList<String> wordsOfCat = words.get(tcat);
		if (wordsOfCat == null) {
			wordsOfCat = new ArrayList<String>();
			words.put(tcat, wordsOfCat);
		}
		wordsOfCat.add(word);
		wordCategories.put(wordKey, tcat);
	}	

	/** Returns a random word of a given lexical category */
	public String generateWord(TokenCategory category) {
		ArrayList<String> wordsOfCat = words.get(category);
		return wordsOfCat.get(prng.nextInt(wordsOfCat.size()));
	}

	/** Returns the type of the word, or TokenCategory.UNKNOWN
		if the word is not in the dictionary
	*/
	public TokenCategory classifyWord(String word) {
		word = word.toLowerCase();
		
		//If the word matches _*, it's a blank...
		boolean isBlank = true;
		for (int c = 0; c < word.length(); ++c) {
			if (word.charAt(c) != '_') {
				isBlank = false;
				break;
			}
		}

		if (isBlank)
			return TokenCategory.Blank;

		if (wordCategories.containsKey(word))
			return wordCategories.get(word);
		else return TokenCategory.Unknown;
	}
}

/**
 This type of exception is thrown if there is an error reading in the
 dictionary file. The error message contains the description of the problem
 @see Dictionary
*/
class DictionaryException extends Exception {
	public DictionaryException(String message) {
		super(message);
	}
}
