/** An applet for program Thermostat */

import java.awt.*;
import java.applet.Applet;
import java.text.*;

public class ThermoApplet extends Applet {
    static boolean outputToggle= false; // = "output should be printed
    //    at each tick of the clock"
    static public int period= 5000;  // Time to wait between events
    
    static SystemClock c;    // The system clock
    static Thread cThread;    // (and its thread)
    
    static OutsideTemperature outTemp; // Simulation of outside temperature
    static Thread outTempThread;  // (and its thread)
    
    static Furnace f;     // Simulation of the furnace
    static Thread fThread;    // (and its thread)
    
    static DesiredTemperature desTemp; // Simulation of desired temperature
    static Thread desTempThread;  // setting (and its thread)
    
    static InsideTemperature inTemp; // Simulation of the inside
    static Thread inTempThread;   // temperature (and its thread)
    
    static String s= "";  // The output String
    static SystemOutput sysout; // Simulate the system output window
    
    // Gives a format for printing decimal numbers.
    public static DecimalFormat decform= new DecimalFormat("000.0000");
    
    /** Called by the browser to start the program */
    public void init() {
        main(null);
    }
    
    /** The method called by the system to start the home heating system simulation. */
    public void main(String[] args) {
        initializeProcesses();
        // Here is the main loop of the simulation. Each iteration
        //    (0) Waits for a clock tick (by calling sleep).
        //    (1) Resets the heat --there's gain due to the furnace being
        //        on (0 if not on)-- and a loss due to the
        //        difference in inside and outside temperatures
        //    (2) Prints a single line of output (using System.out.print..)
        //    if outputToggle is true
        //    (3) Turns the furnace on or off, depending on the inside
        //        temperature and the outside temperature.
        while (true) {
            sleep();
            
            double heatGain= doHeatGain();
            double heatLoss= doHeatLoss();
            if (outputToggle) {
                String s= "" + c.ticks();
                if (f.readValue() == 0)
                    s= s + " furnace off ";
                else s= s + " furnace on ";
                s= s + " Gain: " + decform.format(heatGain) +
                    "  Loss: " + decform.format(-heatLoss) +
                    "  Total change: " + decform.format(heatGain+heatLoss);
                s= s + "  Inside temp: " + decform.format(inTemp.readValue());
                sysout.area.appendText("\n" + s);
            }
            
            // Turn the furnace on or off, if necessary
            if (f.readValue() == 0)
                checkTurnOn();
            else checkTurnOff();
        }
    }
    
    // Create all the processes --clock, furnace, outside temperature,
    // desired temperature, and inside temperature-- and
    // start their threads going
    public static void initializeProcesses() {
        // Create instances of the five processes
        c= new SystemClock(period);
        outTemp= new OutsideTemperature(c, 32);
        f= new Furnace(c);
        desTemp= new DesiredTemperature(c, 68);
        inTemp  = new InsideTemperature(c, 60);
        
        cThread= new Thread(c);
        outTempThread= new Thread(outTemp);
        fThread= new Thread(f);
        desTempThread= new Thread(desTemp);
        inTempThread= new Thread(inTemp);
        
        cThread.setDaemon(true);
        outTempThread.setDaemon(true);
        fThread.setDaemon(true);
        desTempThread.setDaemon(true);
        inTempThread.setDaemon(true);
        
        // Start the five processes
        cThread.start();
        outTempThread.start();
        fThread.start();
        desTempThread.start();
        inTempThread.start();
        
        sysout= new SystemOutput();
    }
    
    /** Wait for the next tick of clock c */
    public static void sleep() {
        synchronized(c) {
            try {
                c.wait();
            }
            catch(InterruptedException ie) {
                System.out.println("Interrupt exception in Thermostat.sleep");
            }
        }
    }
    
    /** If the furnace is on, raise the inside temperature by 1 degree.
        Return the heat gained. */
    public static double doHeatGain() {
        if (f.readValue() == 0)
            return 0;
        double newTemp= inTemp.readValue()+1;
        inTemp.setValue(newTemp);
        return 1;
    }
    
    /** Reduce the inside temperature depending on how cold
        it is outside. Let delta be the amount by which to change
        the temperature:
          delta = (outside temp - inside temp) / 72.
        Add delta to the inside temperature
        Return delta. */
    public static double doHeatLoss() {
        double outT= outTemp.readValue();
        double inT= inTemp.readValue();
        double delta= (outT-inT) / 72.;
        inTemp.setValue(inT + delta);
        return delta;
    }
    
    /** The furnace is on. Turn it off if it should be off. This
       has to be done keeping method checkTurnOn in mind as well.
       For example, if you make this test:
          inside temperature > desired Temperature
       and the corresponding test in method checkTurnOn:
          inside temperature < desired Temperature
       then the furnace may turn on or off at every clock tick,
       which is bad for the furnace. So provide in both methods
       some way to ensure that such quick switching on and off
       does not occur. */
    public static void checkTurnOff() {
        if (inTemp.readValue()  > desTemp.readValue()+1)
            f.turnOff();
    }
    
    /** The furnace is off. Turn it on if it should be on
        See the comment on method checkTurnOff. */
    public static void checkTurnOn() {
        if (inTemp.readValue()  < desTemp.readValue()-1)
            f.turnOn();
    }
    
    /** Paint using Graphics g */
    public void paint(Graphics g) {
        g.drawString("Thermostat!", 30, 30);
    }
}

