/**
 * Just the magnitude aspect of a big integer---no sign, yet.
 * <p>
 * This extends InfiniteDigitArray with sign-oblivious arithmetic and
 * relational operations.
 * <p>
 * When represented as a string, a MyBigMagnitude looks like an unsigned
 * integer.
 */

class MyBigMagnitude extends InfiniteDigitArray implements Comparable {
  // [these constructors are the analogues of the ones in MyBigInteger...]

  public MyBigMagnitude()  { }

  public MyBigMagnitude(int n) {
    this();  // ...redundant, but included for emphasis

    // the standard base-conversion algorithm...
    for (int i = 0; n > 0; i ++, n /= BASE)
      setDigit(i, n % BASE);
  }

  public MyBigMagnitude(String decimal) {
    super(decimal);

    if (decimal.length() == 0)  throw new NumberFormatException();
  }

  //

  public void plus(MyBigMagnitude n) {
    final int w = getHighestOrderNonzero(n);

    for (int i = 0, carry = 0; i <= w + 1; i ++) {
      int  a = getDigit(i) + n.getDigit(i) + carry;

      setDigit(i, a % BASE);
      carry = a / BASE;
    }
  }

  // assumes that n < this...
  public void minusLesserMagnitude(MyBigMagnitude n) {
    final int w = getHighestOrderNonzero(n);

    for (int i = 0, borrow = 0; i <= w; i ++) {
      int  q = getDigit(i) - borrow;
      int  a = q - n.getDigit(i);

      setDigit(i, (BASE + a) % BASE);  // ...i.e., a mod BASE
      borrow = (a < 0) ? 1 : 0;
    }
  }

  protected void timesDigit(byte d) {
    final int w = getHighestOrderNonzero();

    for (int i = 0, carry = 0; i <= w + 1; i ++) {
      int  a = getDigit(i) * d + carry;

      setDigit(i, a % BASE);
      carry = a / BASE;
    }
  }

  public void times(MyBigMagnitude n) {
    final int w = n.getHighestOrderNonzero();

    MyBigMagnitude  sum     = new MyBigMagnitude(0);
    MyBigMagnitude  summand = new MyBigMagnitude(0);

    for (int i = 0; i <= w; i ++) {
      // summand = this * n.getDigit(i) * BASE^i...
      summand.setValue(this);
      summand.timesDigit(n.getDigit(i));
      summand.leftShift(i);
        // [...shifting left i times is equivalent to multiplying by BASE^i.]

      // sum += summand...
      sum.plus(summand);
    }

    setValue(sum);
  }

  //

  /**
   * Returns the decimal representation of the stored magnitude.
   */
  public String toString() {
    if (-1 == getHighestOrderNonzero())  return "0";

    // warning:  assumes BASE == 10...

    return super.toString();
  }

  /**
   * Compares this to o.  It returns -1 if this < o, 0 if this equals o,
   * and +1 if this > o (assuming that o refers to a MyBigMagnitude).
   * <p>
   * [See the standard java.lang.Comparable interface for full detail.]
   * <p>
   * You might think of this as combining equals and lessThan into a single
   * method.
   */
  public int compareTo(Object o) {
    MyBigMagnitude  n = (MyBigMagnitude) o;

    final int i = getHighestOrderDifferenceWith(n);

    if (-1 == i)  return 0;  // ...equal

    return (getDigit(i) < n.getDigit(i)) ? -1 : +1;
  }

  public boolean equals(Object o)
    { return (o instanceof MyBigMagnitude) && 0 == compareTo(o); }

  public Object clone() {
    MyBigMagnitude  n = new MyBigMagnitude();

    n.setValue(this);

    return n;
  }
}
