/**
 * A mutable, arbitrary-precision integer.  An arithmetic operation on a
 * MyBigInteger never results in overflow or "wrap
 * around"---instead, the sequence of digits expands as necessary to
 * accomodate any increase in the stored value's magnitude.  For example,
 *
 * <pre>   MyBigInteger i = new MyBigInteger(65536);
 *
 *   System.out.println(i);  // 65536
 *   i = i.squareOf();
 *   System.out.println(i);  // 4294967296
 *   i = i.squareOf();
 *   System.out.println(i);  // 18446744073709551616
 *   i = i.squareOf();
 *   System.out.println(i);  // 340282366920938463463374607431768211456
 *   i = i.squareOf();
 *   System.out.println(i);  // 115792089237316195423570985008687907853
 *                           //  269984665640564039457584007913129639936
 * 
 *   MyBigInteger j = new MyBigInteger(1);
 *   j.plus(i);
 *   System.out.println(j);  // 115792089237316195423570985008687907853
 *                           //  269984665640564039457584007913129639937</pre>
 * <p>
 * [MyBigInteger is compatible only with itself.]
 */

class MyBigInteger extends MyBigMagnitude implements MyIntegralNumber {
  /**
   * The sign of the stored integer:  -1 if negative, 0 if zero,
   * +1 if positive.
   */ 
  private byte sign;

  //

  /**
   * Constructs a new MyBigInteger object that initially stores
   * the integer value 0.
   */
  public MyBigInteger()  { }

  /**
   * Constructs a new MyBigInteger object that initially stores
   * the integer value i.
   *
   * @param  i  the integer value to initially store in the new object
   */
  public MyBigInteger(int i) {
    super(Math.abs(i));

    setSign((i < 0) ? -1 : (i > 0) ? +1 : 0);
  }

  /**
   * Constructs a new MyBigInteger object that initially stores
   * the integer value represented by the decimal string decimal.
   *
   * @param  decimal  the decimal representation of the integer value to
   *                  initially store in the new object
   */
  public MyBigInteger(String decimal) {
    super(trimLeadingMinus(decimal));

    final boolean isZero          = (-1 == getHighestOrderNonzero());
    final boolean hadLeadingMinus = decimal.trim().startsWith("-");

    setSign(isZero ? 0 : hadLeadingMinus ? -1 : +1);
  }

  //

  final public byte getSign()  { return sign; }

  final public void setSign(byte sign) {
    SanityCheck.assert(sign >= -1 && sign <= +1, "invalid sign");

    this.sign = sign;
  }

  // this variant of setSign allows us to avoid a great deal of tedious
  //  (byte) casting, :)...
  final protected void setSign(int sign)  { setSign((byte) sign); }

  //

  /**
   * Makes this into a copy of n.
   */
  public void setValue(MyBigInteger n)
    { super.setValue(n);  setSign(n.getSign()); }

  public void setToZero()  { super.setToZero();  setSign(0); }

  public Object clone() {
    MyBigInteger  n = new MyBigInteger();

    n.setValue(this);

    return n;
  }

  public MyNumber createCompatible(int i)  { return new MyBigInteger(i); }

  //

  /**
   * Compares this to o.  It returns -1 if this < o, 0 if this equals o,
   * and +1 if this > o (assuming that o refers to a MyBigMagnitude).
   * <p>
   * [See the standard java.lang.Comparable interface for full detail.]
   * <p>
   * You might think of this as combining equals and lessThan into a single
   * method.
   */
  public int compareTo(Object o) {
    MyBigInteger  n = (MyBigInteger) o;

    if (getSign() < n.getSign())  return -1;
    if (getSign() > n.getSign())  return +1;

    return getSign() * super.compareTo(n);
  }

  public boolean equals(Object o)
    { return (o instanceof MyBigInteger) && 0 == compareTo(o); }

  public boolean lessThan(MyNumber n)  { return -1 == compareTo(n); }

  //

  public void minus()  { setSign(-1 * getSign()); }

  public void plus(MyNumber n)  { plus((MyBigInteger) n); }

  public void plus(MyBigInteger n) {
    if (n.getSign() == 0)
      return;

    if (n.getSign() == getSign())
      { super.plus(n);  return; }

    // since the signs are different (either -1 & +1 or +1 & -1), we're
    //  "really" doing subtraction, not addition...

    // compare magnitudes...
    final int comparison = super.compareTo(n);

    if (comparison ==  0)
      // equal magnitudes...
      { setToZero();  return; }

    if (comparison == +1)
      // n is of lesser magnitude---so subtract n...
      { super.minusLesserMagnitude(n);  return; }

    if (comparison == -1) {
      // n is of greater magnitude---so subtract from n...

      MyBigInteger  wasThis = (MyBigInteger) clone();
      setValue(n);
      super.minusLesserMagnitude(wasThis);
    }
  }

  public MyNumber squareOf() {
    MyNumber  n = (MyNumber) clone();

    n.times(n);

    return n;
  }

  public void times(MyNumber n)  { times((MyBigInteger) n); }

  public void times(MyBigInteger n) {
    super.times(n);
    setSign(getSign() * n.getSign());
  }

  public boolean divides(MyIntegralNumber n)
    { return MyNumbers.divides(this, n); }

  //

  public String toString()
    { return ((getSign() < 0) ? "-" : "") + super.toString(); }

  //

  /**
   * Trims the leading minus sign ('-') off of s, if one exists.  Also
   * trims leading and trailing spaces.
   */
  protected static String trimLeadingMinus(String s) {
    s = s.trim();

    if (! s.startsWith("-"))  return s;

    return s.substring(1, s.length()).trim();
  }
}
