/** 
 * Just the digit-storage aspect of a big integer---no arithmetic or sign, yet.
 * <p>
 * Consider the integer 98765003---although we usually don't, we could also
 * write that number as 000098765003, as 000000000000098765003, or even as
 * 00000000000000000000000000000000000000000000000000000098765003.  It looks
 * odd, but, in fact, when we add numbers by hand, we do implicitly "pad" the
 * shorter number with such "leading zeros," in the sense that...
 * <pre>
 *      121110777777321
 *    +        98765003
 *      ---------------
 *      121110876542324
 * </pre>
 * ...is really...
 * <pre>
 *      121110777777321
 *    + 000000098765003
 *      ---------------
 *      121110876542324
 * </pre>
 * Taking this a one step further, we can even think of 98765003 as having
 * "infinitely many" implicit leading zeros, leading us to the idea of an
 * infinite array of digits...
 * <pre>
 *         "infinitely many" leading zeros
 *                        |
 *                        v
 *      ...000000000000000000000000000000000000098765003
 *                                              ^^^^^^^^
 *                                                 |
 *                                finite "suffix" that begins with the
 *                                     highest-order nonzero digit
 * </pre>
 * Of course, concretely, the "infinite" array is really just a finite array
 * (which stores only the "suffix") with the convention that a "get" beyond
 * the end of the suffix array retrieves an implicit 0 and a "set" beyond the
 * end of the suffix array transparently extends it.
 * <p>
 * Why do things this way?  By taking an abstract approach to the digit
 * storage problem, we are able to manipulate digits in a fairly carefree
 * manner, without worrying about the tedius and error-prone steps of
 * checking bounds, expanding storage, etc.  We can get and set digits at
 * will, without asking ourselves "do i need to allocate more storage?", "is
 * this index too big?", ...  Put differently:  InfiniteDigitArray is closer
 * to our "natural" mode of manipulating digits---on paper, we automatically
 * interpret "blanks" as zeros, tack on extra digits at will, and so on;
 * why not operate the same way in our code?
 * <p>
 * [Note:  in keeping with the usual conventions for writing numbers in
 * decimal, when an InfiniteDigitArray is represented as a string,
 * the lowest-order digits appear at the end of the string, and only the
 * "suffix" digits are shown.  (Most of the examples above are atypical; we'd
 * usually just use "98765003".)]
 */

class InfiniteDigitArray {
  /**
   * Every digit is >= 0 and <= BASE.  In this simple implementation,
   * BASE == 10; we're using "good old decimal digits," one decimal digit per
   * array element.
   * <p>
   * [A more sophisticated / efficient implementation would use a larger
   * BASE (as well as a larger element type for the suffix array).]
   */
  final public static int BASE = 10;

  //

  /**
   * Storage for suffix, one digit per array element.
   */
  private byte[] suffix;

  /**
   * The index of the highest-order nonzero digit; -1 if all digits are 0.
   */
  private int highestOrderNonzero;

  //

  /**
   * Constructs a new InfiniteDigitArray whose digits are initially all 0.
   */
  public InfiniteDigitArray() {
    suffix = new byte[0];
    highestOrderNonzero = -1;
  }

  /**
   * Constructs a new InfiniteDigitArray whose digits are initially taken from
   * the given decimal string.
   */
  public InfiniteDigitArray(String decimal) {
    this();

    decimal = decimal.trim();

    // warning:  assumes BASE == 10...

    for (int i = decimal.length(), j = 0; i > 0; i --, j ++) {
      String  decimalDigit = decimal.substring(i - 1, i);
      setDigit(j, Integer.parseInt(decimalDigit));
    }
  }

  //

  /**
   * Gets digit i, where digit 0 is the lowest-order digit.
   * <p>
   * If i > getHighestOrderNonzero() or i < 0, it simply returns 0.
   */
  final public byte getDigit(int i) {
    return (i < 0 || i > getHighestOrderNonzero())
         ? 0 : suffix[i];
  }

  /**
   * Sets digit i to d.
   * <p>
   * Digit storage expands as necessary.
   */
  final public void setDigit(int i, byte d) {
    SanityCheck.assert(d >= 0 && d < BASE, "illegal digit");

    if (i >= suffix.length)
      expandToAccomodate(i);

    suffix[i] = d;

    if (i > highestOrderNonzero)
      highestOrderNonzero = i;

    if (d == 0)
      while (highestOrderNonzero >= 0 && suffix[highestOrderNonzero] == 0)
        -- highestOrderNonzero;
  }
  
  // this variant of setDigit allows us to avoid a great deal of tedious
  //  (byte) casting, :)...
  final protected void setDigit(int i, int d)  { setDigit(i, (byte) d); }

  /**
   * Returns the index of the highest-order nonzero digit.
   * <p>
   * [Returns -1 if all digits are zero.]
   */
  final public int getHighestOrderNonzero()
    { return highestOrderNonzero; }

  final protected int getHighestOrderNonzero(InfiniteDigitArray ida) {
    return Math.max(getHighestOrderNonzero(), ida.getHighestOrderNonzero());
  }

  // 

  /**
   * Expands internal digit storage so that there's at least enough room
   * to store digit i.
   */
  protected void expandToAccomodate(int i) {
    // expand by at least a factor of 2...
    final int newSize = Math.max(i + 1, 2 * suffix.length);

    byte[]  newSuffix = new byte[newSize];

    System.arraycopy(suffix, 0, newSuffix, 0, suffix.length);

    suffix = newSuffix;
  }
  
  //

  /**
   * Shifts all digits k places to the left and fills the vacated locations
   * with zeros.
   */
  public void leftShift(int k) {
    if (k <= 0)  return;

    for (int i = getHighestOrderNonzero(); i >= 0; i --)
      setDigit(i + k, getDigit(i));

    for (int i = k - 1; i >= 0; i --)
      setDigit(i, 0);
  }

  //

  public Object clone() {
    InfiniteDigitArray  ida = new InfiniteDigitArray();

    ida.setValue(this);

    return ida;
  }

  /**
   * Makes this into a copy of ida.
   */
  public void setValue(InfiniteDigitArray ida) {
    suffix = (byte[]) ida.suffix.clone();
    
    highestOrderNonzero = ida.highestOrderNonzero;
  }

  /**
   * Sets all digits to 0.
   */
  public void setToZero()  { highestOrderNonzero = -1; }

  //

  /**
   * Returns the concatenation of the array's suffix digits, from highest-
   * order (at the beginning of the string) down to lowest-order (at the
   * string's end).  The leading zeros are omitted.  [For an all-zero array,
   * this means that the empty string ("") is returned (not "0" !).]
   */
  public String toString() {
    String  s = "";

    for (int i = getHighestOrderNonzero(); i >= 0; i --)
      s += getDigit(i);

    return s;

    // [it would be more efficient to use a StringBuffer, :).]
  }

  public boolean equals(Object o) {
    return (o instanceof InfiniteDigitArray)
        && -1 == getHighestOrderDifferenceWith((InfiniteDigitArray) o);
  }

  /**
   * Returns the index of the highest-order digit at which this and ida
   * differ.  In other words, it returns the largest i such that
   * getDigit(i) != ida.getDigit(i).
   * <p>
   * [It returns -1 if all of the digits are the same.]
   */
  protected int getHighestOrderDifferenceWith(InfiniteDigitArray ida) {
    for (int i = getHighestOrderNonzero(ida); i >= 0; i --)
      if (getDigit(i) != ida.getDigit(i))  return i;

    return -1;
  }

  // ...we'll ignore hashCode, for now...
}
