import java.awt.*;
import java.awt.event.*;

//  GUI for SaM Simulator
public class SaMSimulatorGUI extends Frame implements IDisplay {

	private SaM machine;
	private Panel programPanel;
	private List instructionList;
	private Panel stackPanel;
	private List stack;
	private TextField addrTF;
	private TextField valueTF;
	private TextArea console;
	private List registers;

    private Button rebootButton;
	private Button runButton;
	private Button loadButton;
	private Button stepButton;
	private Button abortButton;

    private Button readButton;
    private Button writeButton;

    private Button powerOffButton;
	
	public SaMSimulatorGUI (SaM m) {
      
		machine = m;

		Font TitleFont = new Font("Helvetica", Font.BOLD, 14);
		setTitle("SaM Console");
		setLayout(new FlowLayout(FlowLayout.CENTER,30,10));

		//Console
		console = new TextArea("Welcome to SaM\n", 10, 40);
		console.setBackground(Color.lightGray);
		add(console);
		
		//Boot panel     
		Panel bootPanel = new Panel();
		bootPanel.setLayout(new GridLayout(2,1,32,8));
		
		rebootButton = new Button("REBOOT");
		rebootButton.setBackground(Color.red);
		rebootButton.setForeground(Color.black);
		bootPanel.add(rebootButton);
		
		powerOffButton = new Button("POWER OFF");
		powerOffButton.setBackground(Color.red);
		powerOffButton.setForeground(Color.black);
		bootPanel.add(powerOffButton);
		
		add(bootPanel);
		
		//Memory panel
		Panel memoryPanel = new Panel();
		memoryPanel.setLayout(new BorderLayout(10,10));
		memoryPanel.setBackground(Color.lightGray);

		Panel labelPanel = new Panel();
		labelPanel.setLayout(new FlowLayout());
		Label ML = new Label("Memory", Label.CENTER);
		ML.setFont(TitleFont);
		ML.setForeground(Color.red);
		labelPanel.add(ML);
		memoryPanel.add("North", labelPanel);
		
		Panel buttonPanel = new Panel();
		buttonPanel.setLayout(new GridLayout(2,1,4,4));
		buttonPanel.setForeground(Color.black);
		readButton = new Button("READ");
		readButton.setBackground(Color.yellow);
		buttonPanel.add(readButton);
		writeButton = new Button("WRITE");
		writeButton.setBackground(Color.green);
		buttonPanel.add(writeButton);     
		memoryPanel.add("East", buttonPanel);
		
		Panel valuePanel = new Panel();
		valuePanel.setLayout(new GridLayout(2,1,4,4));
		Label val = new Label ("VALUE", Label.CENTER);
		val.setForeground(Color.red);
		valuePanel.add(val);
		valueTF = new TextField(6); 
		valuePanel.add(valueTF);
		memoryPanel.add ("Center", valuePanel);
		   
		Panel addrPanel = new Panel();
		addrPanel.setLayout(new GridLayout(2,1,4,4));
		Label add = new Label ("ADDRESS", Label.CENTER);
		add.setForeground(Color.red);
		addrPanel.add (add);
		addrTF = new TextField(6); 
		addrPanel.add(addrTF);
		memoryPanel.add ("West", addrPanel);
  
		add(memoryPanel);
		
		//Stack panel
		stackPanel = new Panel();
		stackPanel.setLayout(new BorderLayout(5,5));
		stackPanel.setForeground(Color.black);
		stackPanel.setBackground(Color.lightGray);
		
		Label SL = new Label ("Stack", Label.CENTER);
		SL.setFont(TitleFont);
		SL.setForeground(Color.red);
		stackPanel.add("North", SL);
		
		stack = new List(16, false);
		stack.setForeground(Color.black);
		stackPanel.add("Center", stack);
		
		add(stackPanel);
      		
		//Program panel   
		programPanel = new Panel();
		programPanel.setLayout(new BorderLayout(30,30));
		programPanel.setForeground(Color.black);
		programPanel.setBackground(Color.lightGray);
		
		Label PL = new Label ("Program", Label.CENTER);
		PL.setFont(TitleFont);
		PL.setForeground(Color.red);
		programPanel.add("North", PL);
		
		instructionList = new List(8, false);
		instructionList.setForeground(Color.black);
		programPanel.add("Center", instructionList);
		
		Panel runnerPanel = new Panel();
		runnerPanel.setLayout(new GridLayout(3,1,8,4));
		
		loadButton = new Button(" LOAD ");
		loadButton.setForeground(Color.black);
		loadButton.setBackground(Color.red);
		runnerPanel.add(loadButton);
		
		stepButton = new Button(" STEP ");
		stepButton.setForeground(Color.black);
		stepButton.setBackground(Color.yellow);
		stepButton.setEnabled(false);
		runnerPanel.add(stepButton);
		
		runButton = new Button("RUN");
		runButton.setForeground(Color.black);
		runButton.setBackground(Color.green);
		runButton.setEnabled(false);
		runnerPanel.add(runButton);
		programPanel.add("East", runnerPanel);
		
		add(programPanel);
		
		// Register panel
		Panel registerPanel = new Panel();
		registerPanel.setLayout(new BorderLayout(5,5));
		registerPanel.setForeground(Color.black);
		registerPanel.setBackground(Color.lightGray);
		
		Label RL = new Label ("Registers", Label.CENTER);
		RL.setFont(TitleFont);
		RL.setForeground(Color.red);
		registerPanel.add("North", RL);
		
		registers = new List(5, false);
		registers.setForeground(Color.black);
		
		registers.addItem("PC:  " + machine.getPC() );
		registers.addItem("FBR: " + machine.getFBR() ); 
		registers.addItem("SP:  " + machine.getSP() );
		registers.addItem("HP:  " + machine.getHP() );
		
		registerPanel.add("South", registers);
		
		add(registerPanel);
		
        // Colors for the main frame
		setForeground(Color.black);
  		setBackground(Color.cyan);

        // Add the listeners
        addListeners();

        // show it all
        setSize(750, 500);
        setVisible(true);
    } 

    public void println(String s) { console.append(s + "\n"); }

    public void print(String s) { console.append(s); }

    public void runDone() { //call back from SaM when run is done
		//Terminate.setVisible(false);
		update();
		runButton.setLabel("RUN"); //change button back to RUN
		runButton.setBackground(Color.green);
		loadButton.setEnabled(true);
		stepButton.setEnabled(true);
		instructionList.makeVisible(machine.getPC()); //show current instruction    
		instructionList.select(machine.getPC()); //hilight it
		programPanel.repaint();
    }

    public void consoleClear() {
		console.selectAll();
		console.replaceRange("Rebooting SaM \n", console.getSelectionStart(), 
		                    console.getSelectionEnd());
    }
 
    public void update() {
		//update registers
		registers.removeAll();
		registers.addItem("PC:  " + machine.getPC() );
		registers.addItem("FBR: " + machine.getFBR() ); 
		registers.addItem("SP:  " + machine.getSP() );
		registers.addItem("HP:  " + machine.getHP() );
		
		//update stack
		stack.removeAll();
		int stackTop = machine.getSP();
		for (int i=0; i< stackTop; i++) {
		       stack.addItem( i + ": " + machine.getStackElement(i) ); }
		if (stackTop!=0) 
		       stack.select(stackTop-1);
		instructionList.select(machine.getPC()); //hilight it
		programPanel.repaint();
    }

    private String getFileName() {
		FileDialog fd = new FileDialog(this,"JRM Command File", FileDialog.LOAD);
		
		fd.setResizable(true);
		fd.setSize(500,500);
		fd.setBackground(Color.white);
		fd.show();
		
		// get the name of the file chosen
		String s = fd.getFile();
		String dir = fd.getDirectory();
		return dir + s;
    }

    private void addListeners() {

        rebootButton.addActionListener( new ActionListener() {
            public void actionPerformed(ActionEvent evt) {
				machine.reboot();
				update();
				//ShowRegs();
				if (instructionList.getItemCount() != 0) { //if there is an instruction at 0
				    instructionList.makeVisible(machine.getPC()); //show start of program
				    instructionList.select(machine.getPC());
	            }	
            }
        });

        loadButton.addActionListener( new ActionListener() {
            public void actionPerformed(ActionEvent evt) {
				machine.reboot();
				String filename = getFileName();
				
				if (filename == null) {
	                println("No SaM Command File Selected");
				    return;
			    } else {
					println("Loading SaM Commands from file " + filename);
					     
					instructionList.removeAll();//clear Program display
					
					if (filename != null) {
						int PSize = machine.load(new CS211In(filename));
						
						//update List of instructions in display    
						for (int i = 0; i < PSize; i++) {
							if (machine.getInstruction(i) != null)
							    instructionList.addItem(i + ": " + machine.getInstruction(i), i);
							else if ((i != 0) && (machine.getInstruction(i-1) != null))
							    //this is first empty slot
							    instructionList.addItem(".........",i); //at end of program
						}
						
						machine.setPC(0);
						instructionList.makeVisible(machine.getPC()); //show start of program
						instructionList.select(machine.getPC());
						stepButton.setEnabled(true);
						runButton.setEnabled(true);
					}
			    }
            }
        });

        runButton.addActionListener( new ActionListener() {
            public void actionPerformed(ActionEvent evt) {
                String arg = evt.getActionCommand();
				if (arg.equals ("RUN")) {
				        
					machine.start();//create thread and run
					runButton.setLabel("ABORT ");
					runButton.setBackground(Color.red);
					loadButton.setEnabled(false);
					stepButton.setEnabled(false);
					programPanel.repaint(); 
					//returns before program terminates!!!
				}
				
				if (arg.equals ("ABORT ")) {
				  
					machine.stop(); //call stop routine
					runButton.setLabel("RUN");
					runButton.setBackground(Color.green);
					loadButton.setEnabled(true);
					stepButton.setEnabled(true);
					programPanel.repaint();
				}
            }
        });

        stepButton.addActionListener( new ActionListener() {
            public void actionPerformed(ActionEvent evt) {
				machine.step(); //perform operation
				update();
				instructionList.makeVisible(machine.getPC()); //show next instruction
				instructionList.select(machine.getPC());      //hilight it
            }
        });

        readButton.addActionListener( new ActionListener() {
            public void actionPerformed(ActionEvent evt) {
				int address;
				String s = addrTF.getText();
				try  {
	                address = Integer.parseInt(s, 10); //try conversion
				    valueTF.setText(Integer.toString(machine.getStackElement(address)));
				} catch (NumberFormatException ex) {  //catch bad strings here
				    println("Bad address: " + s);
				} catch (ArrayIndexOutOfBoundsException ex) { //catch illegal address
				    println("Bad address: " + ex.getMessage());
				}
            }
        });

        writeButton.addActionListener( new ActionListener() {
            public void actionPerformed(ActionEvent evt) {
				int address;
				int val;
				String s1 = addrTF.getText();
				String s2 = valueTF.getText();
				
				try {
	                address = Integer.parseInt(s1, 10); //try conversion
				    val     = Integer.parseInt(s2, 10);
				    machine.setStackElement(address,val);
				    println("Set location " + address + " to " + val);
				} catch (NumberFormatException ex) { //catch bad strings here
				    println("Bad input: " + ex.getMessage());
				} catch (ArrayIndexOutOfBoundsException ex) { //catch illegal address
				    println("Bad address: " + ex.getMessage());
				}
            }
        });

        powerOffButton.addActionListener( new ActionListener() {
            public void actionPerformed(ActionEvent evt) {
				machine.powerOff();
				setVisible(false);
				dispose();
                System.exit(0);
            }
        });
    }
}