import java.io.*;
//********************************Instructions******************************************

//This is the abstract superclass for all instructions

abstract class Instruction implements I_Instruction {
    abstract public void execute(SaM machine);  // Strictly not necessary.

    public void readOperand(CS211InInterface inf)  {}

    public void translate(SymbolTable ST, SaM m) {}

    protected String mnemonic() { //the opcode
		String s = this.getClass().getName();
		return s.substring(0, s.length() - ("Inst".length()));
    }

    public String toString() {
        return mnemonic();
    }
}

//This is the class for all instructions with one operand.
//The operand can be a word or an integer. 
abstract class InstructionWithOperand extends Instruction {
    protected String dest = new String("");
    protected int operand;

    public String toString() {
		if (dest.equals("")) return mnemonic() + " " + operand;
		else return mnemonic() + " " + dest + " //address:" + operand;
    }

    public String getDest() {//for debuggers
        return dest;
    }

    public void readOperand(CS211InInterface inf) {
      if (inf.peekAtKind() == inf.WORD)
          dest = new String(inf.getWord());
      else if (inf.peekAtKind() == inf.INTEGER)
          operand = inf.getInt();
    }

    public void translate(SymbolTable ST, SaM m) {
        //if we had a symbolic operand
        if (! dest.equals("")) {
            //translate
            operand = ST.lookup(dest);
            if (operand == -1)  m.println("Undefined name " + dest);
        }
    }
}


//********************************Opcodes******************************************

class ADDInst extends Instruction {
    public void execute(SaM machine) {
        int top =  machine.pop();
        int below =  machine.pop();
        machine.push(below + top);
        machine.incPC();
    }
}

class SUBInst extends Instruction {
    public void execute(SaM machine) {
	    int top = machine.pop();
	    int below = machine.pop();
	    machine.push(below - top);
	    machine.incPC();
    }
}

class TIMESInst extends Instruction {
    public void execute(SaM machine) {
	    int top = machine.pop();
	    int below = machine.pop();
	    machine.push(top * below);
	    machine.incPC();
    }
}

class DIVInst extends Instruction {
    public void execute(SaM machine) {
	    int top = machine.pop();
	    int below = machine.pop();
	    machine.push(below / top);
	    machine.incPC();
    }
}

class GREATERInst extends Instruction {
    public void execute(SaM machine) {
	    int top = machine.pop();
	    int below = machine.pop();
	    machine.push((below > top) ? 1 : 0);
	    machine.incPC();
    }
}

class LESSInst extends Instruction {
    public void execute(SaM machine) {
	    int top = machine.pop();
	    int below = machine.pop();
	    machine.push((below < top) ? 1 : 0);
	    machine.incPC();
    }
}

class EQUALInst extends Instruction {
    public void execute(SaM machine) {
	    if (machine.pop() == machine.pop())
	      machine.push(1);
	    else 
	      machine.push(0);
	    machine.incPC();
    }
}

class ORInst extends Instruction {
    public void execute(SaM machine) {
	    int top = machine.pop();
	    int below = machine.pop();
	    machine.push(((top != 0) || (below != 0)) ? 1 : 0);
	    machine.incPC();
    }
}

class ANDInst extends Instruction {
    public void execute(SaM machine) {
	    int top = machine.pop();
	    int below = machine.pop();
	    machine.push((top == 1) && (below == 1) ? 1 : 0);
	    machine.incPC();
    }
}

class NOTInst extends Instruction {
    public void execute(SaM machine) {
	    machine.push((machine.pop() == 0) ? 1 : 0);
	    machine.incPC();
    }
}


class PUSHIMMInst extends InstructionWithOperand {
    public void execute(SaM machine) {
		machine.push(operand);
		machine.incPC();
    }
}


class DUPInst extends Instruction {
    public void execute(SaM machine) {
	    int top = machine.pop();
	    machine.push(top);
	    machine.push(top);
	    machine.incPC();
    }
}

class SWAPInst extends Instruction {
    public void execute(SaM machine) {
	    int top = machine.pop();
	    int below = machine.pop();
	    machine.push(top);
	    machine.push(below);
	    machine.incPC();
    }
}

class POPSPInst extends Instruction {
    public void execute(SaM machine) {
	    machine.setSP(machine.pop());
	    machine.incPC();
    }
}

class PUSHSPInst extends Instruction {
    public void execute(SaM machine) {
	    machine.push(machine.getSP());
	    machine.incPC();
    }
}

class ADDSPInst extends InstructionWithOperand {
    public void execute(SaM machine) {
	    machine.setSP(machine.getSP() + operand);
	    machine.incPC();
    }
}

class PUSHFBRInst extends Instruction {
    public void execute(SaM machine) {
        machine.push(machine.getFBR());
        machine.incPC();
    }
}

class POPFBRInst extends Instruction {
    public void execute(SaM machine) {
	    machine.setFBR(machine.pop());
	    machine.incPC();
    }
}

class PUSHINDInst extends Instruction {
    public void execute(SaM machine) {
	    machine.push(machine.getStackElement(machine.pop()));
	    machine.incPC();
    }
}

class STOREINDInst extends Instruction {
    public void execute(SaM machine) {
	    int top = machine.pop();
	    int below = machine.pop();
	    machine.setStackElement(below, top);
	    machine.incPC();
    }
}

class PUSHOFFInst extends InstructionWithOperand {
    public void execute(SaM machine) {
	    machine.push(machine.getStackElement(machine.getFBR() + operand));
	    machine.incPC();
    }
}

class STOREOFFInst extends InstructionWithOperand {
    public void execute(SaM machine) {
	    machine.setStackElement(machine.getFBR() + operand, machine.pop());
	    machine.incPC();
    }
}

class JUMPInst extends InstructionWithOperand {
    public void execute(SaM machine) {
        machine.setPC(operand);
    }
}

class JUMPCInst extends InstructionWithOperand {
    public void execute(SaM machine) { 
	    if (machine.pop() == 1)
	      machine.setPC(operand);
	    else 
	      machine.incPC();
    }
}

class JUMPINDInst extends Instruction {
    public void execute(SaM machine) {
	    int retAddress = machine.pop();
	    if (machine.getInstruction(retAddress - 1) instanceof JSRInst) //method call
			machine.println("Returning from " +
			                ((JSRInst)(machine.getInstruction(retAddress -1))).getDest());
	    machine.setPC(retAddress);
    }
}

class JSRInst extends InstructionWithOperand {
    public void execute(SaM machine) {
	    //print trace information
	    machine.print("Method call:" + dest + "(");
	    int first = machine.getFBR() + 1;
	    int last =  machine.getSP() - 1;
	    if (last < first) //no arguments
	        machine.println(" )");
	    else {
	      for (int i = first; i < last; i++) 
	          machine.print(machine.getStackElement(i) + ",");
	      machine.println(machine.getStackElement(last) + ")");
	    }
	    machine.push(machine.getPC() + 1);
	    machine.setPC(operand);
    }
}

class JSRINDInst extends Instruction {
    public void execute(SaM machine) {
	    int top = machine.pop();
	    machine.push(machine.getPC() + 1);
	    machine.setPC(top);
    }
}

class MALLOCInst extends InstructionWithOperand {
    public void execute(SaM machine) {
	    machine.push(machine.getHP()); 
	    machine.setHP(machine.getHP() - operand);
	    machine.incPC();
    }
}

class READInst extends Instruction {
    private static BufferedReader in = 
                        new BufferedReader(new InputStreamReader(System.in));

    public void execute(SaM machine) {
        try {
			machine.println("Enter an integer in console");
			System.out.print("SaMmy needs int: ");
			System.out.flush();
			machine.push(Integer.parseInt(in.readLine()));
			machine.incPC();
        } 
        catch (Exception e) {
            System.out.println("Error: need an integer");
        }
    }
}

class PRINTInst extends Instruction {
    public void execute(SaM machine) {
        System.out.println("SaMmy says: "+ machine.pop()); 
        machine.incPC();
    }
}

class STOPInst extends Instruction {
    public void execute(SaM machine) {
        machine.halt();
    }
}
