//Another implementation of Ipuzzle interface
//It uses a row-major encoding of the puzzle configuration
//using 9 to encode the missing tile
//eg. the following configuration
//   1 4 5
//   2 3 6
//   7   8
//is encoded as 897632541.
//In other words, tile in position (r,c) becomes 
//digit in position (r*cols+c)

class RowMajorPuzzle implements IPuzzle{

 protected static final int rows = 3;
 protected static final int cols = 3;
 protected static final int base = 10;
 protected static final int emptyNumber = 9; 

 protected int puzzleState = -1; //encoding of configuration 

 //***********public methods******************************

  // create puzzle in sorted state
 public RowMajorPuzzle() {
   puzzleState = 0;
   
   for (int i = 1; i <= rows*cols; i++) 
     puzzleState = puzzleState + encode(i,i-1);
 }

  //return number of rows 
 public int rows() {
  return rows;
 }

  //return number of columns
 public int cols() {
  return cols;
 }

  //overrides method in Object
 public Object clone() {
  RowMajorPuzzle nP = new RowMajorPuzzle();
  nP.puzzleState = puzzleState;   
  return nP;
 }

  //overrides method in Object
 public boolean equals(Object o) {
  if (! (o instanceof RowMajorPuzzle)) return false;
  RowMajorPuzzle nP = (RowMajorPuzzle)o;
  if (puzzleState == nP.puzzleState) return true;
  else return false;
 }

  //return true if puzzle is in sorted state
  public boolean isSorted() {
    for (int i = 0; i < rows; i++)
      for (int j = 0; j < cols; j++)
       if (tile(i,j) != i*cols + j + 1) return false;
    return true;
  }

 public   void scramble() {
   puzzleState = 0;

   for (int i = 1; i <= rows*cols; i++) 
     puzzleState = puzzleState + encode(i,i-1);

   String M =  "SESENNWSESSENNW";
   for (int i = 0; i < M.length(); i++)
       move(M.charAt(i));
 }

  //is row i, column j empty? 
 public boolean isEmpty(int i, int j) {
 return (digit(i*cols + j) == emptyNumber);
 }


 public   int tile(int r, int c) {
   return digit(r*cols + c);
 }

 public   boolean move(char c) {
   //Find location of empty slot
   int zPos = getDigitForEmpty();
   int tPos = getDigitForMovedTile(zPos,c);
   if (tPos == -1) return false; //illegal move
   else 
       {int tileNumber  = digit(tPos);
        puzzleState = puzzleState 
                      - encode(tileNumber,tPos)+encode(tileNumber,zPos) 
                      - encode(emptyNumber,zPos)+encode(emptyNumber,tPos); 
        return true;
       }
 }
 //***********private methods******************************
 private   int digit(int position) {
   return puzzleState/power(base,position)%base;
 }
 private   int encode (int tile, int position){ 
   return tile*power(base,position);
 }
 private   int power (int b, int p) {
  int value = 1;
  for (int i = 0; i < p; i++) 
     value = value*b;
  return value;
 }
 private   int getDigitForEmpty() {
   int i = 0;
   while (i < rows*cols && digit(i) != emptyNumber) 
     i++;
   return i;
 }
 private   int getDigitForMovedTile(int zPos, char c) {
   //Find digit for tile being moved
   switch (c) {
   case 'N' : //make sure empty slot is not at the bottom
     if (zPos >= (rows-1)*cols) return -1;
     else return zPos + cols;
   case 'S' : //make sure empty slot is not at the top
     if (zPos < cols) return -1;
     else return zPos - cols;
   case 'E' : //make sure empty slot is not at the left
     if (zPos % cols == 0) return -1;
     else return zPos - 1;
   case 'W' :  //make sure empty slot is not at the right
     if ((zPos + 1) % cols == 0) return -1;
     else return zPos + 1;
   default: return -1;
   }
 }
}

