/* TemperatureModel.java
 *
 * Walker M. White
 * September 22, 2002
 */
package converter;

import java.awt.*;
import java.text.*;
import javax.swing.*;

/** 
 * This is the MODEL that holds our temperature values and converts
 * between them for us.  We separate this from the view so that we
 * can restore the old state whenever a user enters bad data.  If the
 * view took care of everything, we would not be able to undo bad
 * input.
 *  
 * At any given time, this model should be <i>consistent</i>.  That is, 
 * the farenheit and the centigrade value should never get out of synch.
 * Therefore, whenever the farenheit value is changed, the centigrade
 * value must be changed immediately as well.  This is a prime example
 * of why you should NEVER make fields public, but use get and set 
 * methods instead.
 *
 * In order to ensure consistency, we will only have one field: the 
 * farenheit field.  The centigrade field is computed on the fly 
 * whenever we need it.
 * 
 * Note that the model NEVER talks to the controller or the view.  The
 * model is an entirely passive object; the controller does all of the
 * talking.  This allows for maximum code reuse.
 *  
 * @author  Walker M. White
 * @version 9/22/2002
 */
public class TemperatureModel {
  
    /** Number of digits of precision to use in displaying temperature. */
    private static int PRECISION = 2;

    /** Temperature in degrees farenheit */
    private double farenheit;
 
    /** Formatter to convert temperature to a string */
    private NumberFormat format;
 
    /** Constructor: a new temperature value at 32F */
    public TemperatureModel() {
        this(32);
    }

    /** 
     * Constructor: a new temperature value at given farenheit.
     */
    public TemperatureModel(double farenheit) {
        setFarenheit(farenheit);
  
        // Text formatting (set # of decimal places).
        format = NumberFormat.getInstance();
        format.setMinimumFractionDigits(PRECISION);
        format.setMaximumFractionDigits(PRECISION);
    }

    /** 
     * Yields: the current farenheit.
     */
    public double getFarenheit() {
        return farenheit;
    }

    /** 
     * Yields: the current centigrade.
     */
    public double getCentigrade() {
        return (farenheit-32)*(5.0/9.0);
    }
    
    /** 
     * Yields: the current farenheit as a String.  
     * The precision is defined by the formatter.
     */
    public String getFarenheitAsString() {
        return format.format(farenheit);
    }

    /** 
     * Yields: the current centigrade as a String.  
     * The precision is defined by the formatter.
     */
    public String getCentigradeAsString() {
        return format.format(getCentigrade());
    }

    /** 
     * Sets the temperature to the specified farenheit value.
     */
    public void setFarenheit(double farenheit) {
        this.farenheit = farenheit;
    }


    /** 
     * Sets the temperature to the specified centigrade value.
     */
    public void setCentigrade(double centigrade) {
        farenheit = (centigrade*9.0/5.0)+32;
    }
}

