import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

/** An instance provides the GUI and the board on which the game is played. */
public class JManGUI implements ActionListener{
    
    
    /** Constants that indicate the next direction in which JMan should move */
    /** Jman should move up. */
    public static final int UP= 1;
    /** Jman should move down. */
    public static final int DOWN= 2;
    /** Jman should move left. */
    public static final int LEFT= 3;
    /** Jman should move right. */
    public static final int RIGHT= 4;
    
    /** Window for game with graphics and buttons. */
    public JFrame frame= new JFrame("J*Man!!!"); // The GUI window
    
    // The panel that contains the board of the game. Class JManPanel is an "inner class", defined below.
    private JManPanel panel;                     
    
    // Buttons on the GUI
    private JButton bUp= new JButton("Up");
    private JButton bDown= new JButton("Down");
    private JButton bLeft= new JButton("Left");
    private JButton bRight= new JButton("Right");
    private JButton bnewGame= new JButton("New Game");
    
    /** The next direction in which JMan should move.
      one of Up, Down, Left, Right */
    private int nextJManDirection; 
    
    // Box to contain the direction buttons
    private Box buttonBox= new Box(BoxLayout.X_AXIS);
    
    // The box to contain buttonBox and the instructions
    Box instructBox= new Box(BoxLayout.Y_AXIS);
    
    private int height= 20;  // height of the game board in tiles.
    private int width= 20;   // width of the game board in tiles.
    private int tileWidth= 16;  // width of a tile in pixels.
    private int tileHeight= 16; // height of a tile in pixels.
    
    private Piece[][] board; // 2-d array of Pieces that makes up the game.
    
    private JMan jMan; //the J*Man piece in this game board.
    
    /** = the next direction in which JMan should move: one of
      the constants gui.Up, JManGUI.Down, JManGUI.Left, and Ma.,Right.
      */
    public int nextJManDirection() {
        return nextJManDirection;
    }
    
    /** = Change the next direction in which JMan should move to d.
      Precondition: d is one of the constants JManGUI.Up, JManGUI.Down,
      JManGUI.Left, and Ma.,Right.
      */
    public void changeJManDirection(int d) {
        nextJManDirection= d;
    }
    
    /** Start a game 20x20 game wih 10 walkers, 10 pillars, and 20 blocks. */
    public static void main(String[] pars) {
        JManGUI m= new JManGUI();
    }
    
    /** = the board of pieces that make up the game */
    public Piece[][] getBoard() {
        return board;
    }
    
    /** Constructor: a default 20 x 20 game with 10 walkers, 10 pillars,
      and 20 blocks placed randomly throughout the game board. */
    public JManGUI(){
        this(20, 20, 20, 10, 10);
    }
    
    /** Constructor: a game with an h x w game board with bl blocks,
      wa walkers, and pi pillars. 
      J#Man is at position (0, 0), and all other pieces are placed randomly.
      Precondition: number of pieces specified is <= h*w. */
    public JManGUI(int h, int w, int bl, int wa, int pi){
        this.height= h;
        this.width= w;
        
        // Set the preferred dimensions of the buttons
        Dimension buttondim= new Dimension(width*tileWidth/4,27);
        bUp.setPreferredSize(buttondim);
        bDown.setPreferredSize(buttondim);
        bLeft.setPreferredSize(buttondim);
        bRight.setPreferredSize(buttondim);
        bnewGame.setPreferredSize(new Dimension(width*tileWidth/2,27));
        
        // Add the direction buttons to buttonBox and set the buttonBox alignment
        buttonBox.add(bUp);
        buttonBox.add(bDown);
        buttonBox.add(bLeft);
        buttonBox.add(bRight);
        buttonBox.setAlignmentX(0);
        
        // Register this as an action listener for all buttons.
        addListeners();
        
        // Set up the game board.
        panel= new JManPanel(this);
        
        // Place the direction buttons and the instructions into instructBox and
        // set its alignment.
        instructBox.setAlignmentX(0);
        instructBox.add(buttonBox);
        addInstructions(instructBox);
        
        // Create the game board, put J*Man in (0,0), and
        // put the rest of the pieces randomly on the JManGUI.
        board= new Piece[width][height];
        putNew(1, 0, 0);
        putOnGUI(bl, wa, pi);
        
        // Put the board and buttons and instructions into the frame.
        frame.getContentPane().add(bnewGame, BorderLayout.NORTH);
        frame.getContentPane().add(panel, BorderLayout.CENTER);
        frame.getContentPane().add(instructBox, BorderLayout.SOUTH);
        
        frame.pack();
        frame.setLocation(5,30);
        frame.setResizable(false);
        frame.setVisible(true);
    }
    
    /** Add to instructBox the rules of the game (a sequence of JLabels). */
    private void addInstructions(Box instructBox) {
        instructBox.add(new JLabel(" Use the four buttons to direct J*Man (the star-", SwingConstants.LEFT));
        instructBox.add(new JLabel(" like piece) to capture the other colored pieces.", SwingConstants.LEFT));
        instructBox.add(new JLabel(" J*Man can capture: ", SwingConstants.LEFT));
        instructBox.add(new JLabel("    a green piece if he is yellow,", SwingConstants.LEFT));
        instructBox.add(new JLabel("    a yellow piece if he is red, ", SwingConstants.LEFT));
        instructBox.add(new JLabel("    a red piece if he is green.", SwingConstants.LEFT));
        instructBox.add(new JLabel(" Walkers (triangles) wander randomly about.", SwingConstants.LEFT));
        instructBox.add(new JLabel(" Pillars (circles) change color occasionally.", SwingConstants.LEFT));
        instructBox.add(new JLabel(" Nothing can enter a block (white square).", SwingConstants.LEFT));
        instructBox.add(new JLabel(" Be careful. With patience, you can always capture ", SwingConstants.LEFT));
        instructBox.add(new JLabel(" a pillar, but capturing all walkers requires ", SwingConstants.LEFT));
        instructBox.add(new JLabel(" thinking ahead. Good Luck!", SwingConstants.LEFT));
    }
    
    /** Put bl block, wa walkers, and pi pillars randomly on the JManGUI board
      Precondition. The board must have enough empty spaces for all of them. */
    private void putOnGUI(int bl, int wa, int pi) {
        // Put the blocks on the JManGUI.
        int k= 0;
        // invariant: k blocks have been added.
        while (k < bl) {
            int xx= Piece.rand(0, width-1);
            int yy= Piece.rand(0, height-1);
            if(isEmpty(xx, yy)){
                putNew(Piece.BLOCK, xx, yy);
                k= k+1;
            }
        }
        
        // Put the walkers on the JManGUI.
        k= 0;
        // invariant: k walkers have been added.
        while (k < wa) {
            int xx= Piece.rand(0, width-1);
            int yy= Piece.rand(0, height-1);
            if(isEmpty(xx, yy)){
                putNew(Piece.WALKER, xx, yy);
                k= k+1;
            }
        }
        
        // Put the pillars on the JManGUI.
        k= 0;
        // invariant: k pillars have been added.
        while(k < pi){
            int xx= Piece.rand(0, width-1);
            int yy= Piece.rand(0, height-1);
            if(isEmpty(xx, yy)){
                putNew(Piece.PILLAR, xx, yy);
                k= k+1;
            }
        }
    }
    
    /** If (x, y) is on the board and empty, create a new piece of type t;
      put it in location (x, y) of the board; and,
      if the new piece is J*Man, store it in field jMan.
      Precondition: t is one of the piece constants in class Piece.*/
    public void putNew(int t, int x, int y){
        // complete this method and delete this comment
        
        
        
        
        
        
        
        
        
        
    }
    
    /** = "(x, y) is on the board". */
    public boolean isOnBoard(int x, int y){
        return x > -1 && y > -1 && x < width && y < height;
    }
    
    /** = "(x, y) is on the board and does not contain a piece". */
    public boolean isEmpty(int x, int y){
        return isOnBoard(x,y) && board[x][y] == null;
    }
    
    /** = the Piece at position (x, y) of the board
          (null if (x, y) is outside the board or contains null). */
    public Piece pieceAt(int x, int y){
        if(isOnBoard(x,y)){
            return board[x][y];
        }
        return null;
    }
    
    /** Move the Piece at (fromX, fromY) to (toX, toY) on the board,
      changing the position at (fromX, fromY) to null;
      change the position in the Piece accordingly.
      The piece originally in (toX, toY) is permanently deleted.
      Precondition:
      1. (toX, toY) is on the board.
      2. The move is allowed by the game. */
    public void move(int fromX, int fromY, int toX, int toY) {
        board[toX][toY]= board[fromX][fromY];
        board[fromX][fromY]= null;
        board[toX][toY].setX(toX);
        board[toX][toY].setY(toY);
    }
    
    /** Make every piece on the board act once, with JMan acting first.
      When all have acted, reset all their has-acted flags to false.
      Precondition: the has-acted flag is false for all pieces on the board. */
    public void act(){
        jMan.act();
        jMan.setActed(true);
        
        // Make every other piece act.
        for (int i= 0; i < width; i= i+1){
            for (int j= 0; j < height; j= j+1){
                Piece p= board[i][j];
                if (p != null  &&  !p.hasActed()){
                    p.act();
                    p.setActed(true);
                }
            }
        }
        
        // Set all the act flags to false.
        for (int i= 0; i < width; i= i+1){
            for (int j= 0; j < height; j= j+1){
                if (board[i][j] != null){
                    board[i][j].setActed(false);
                }
            }
        }
    }
    
    /** Deegister this object as a lisenter for the five buttons */
    public void removeListeners() {
        bUp.removeActionListener(this);
        bDown.removeActionListener(this);
        bLeft.removeActionListener(this);
        bRight.removeActionListener(this);
        bnewGame.removeActionListener(this);
    }
    
    /** Register this object as a listener for the five buttons */
    public void addListeners() {
        bUp.addActionListener(this);
        bDown.addActionListener(this);
        bLeft.addActionListener(this);
        bRight.addActionListener(this);
        bnewGame.addActionListener(this);
    }
    
    /** Process a button push and then repaint the game board.
      Deregister this as a button listener of all buttons while the processing is going on
         (and reregister it when done)
      If the button was newGame, open a dialog and ask whether a new game
      is desired and act accordingly.
      
      If the button was bUp, bDown, bLeft, or bRight, save the direction for
      processing by JMan in field nextJManDirection, make JMan act, and
      then make all other pieces act.
      */
    public void actionPerformed(ActionEvent e){
        removeListeners();
        if (e.getSource() == bnewGame){
            if (JOptionPane.showConfirmDialog(frame, "Start a new game of the standard size?","New Game?",
                                              JOptionPane.OK_CANCEL_OPTION) == JOptionPane.OK_OPTION){
                new JManGUI();
                panel.repaint();//repaint the game board
                addListeners();
                return;
            }
        }
        /** Set field nextJManDirection to the direction in which JMan should move
            -- one of the JMAnGUI constants Up, Down, Left, Right */
        if (e.getSource() == bUp){
            nextJManDirection= UP;
        } else if (e.getSource() == bDown){
            nextJManDirection= DOWN;
        } else if (e.getSource() == bLeft){
            nextJManDirection= LEFT;
        } else if (e.getSource() == bRight){
            nextJManDirection= RIGHT;
        } else {
            throw new RuntimeException("In JMan. Unknown button was pressed");
        }
        
        act();
        
        panel.repaint();
        addListeners();
    }
    
    
    
    /* Inner class to take care of the graphics. */
    private class JManPanel extends JPanel{
        private JManGUI m;
        
        /* Constructor: an instance with JManGUI m. */
        public JManPanel(JManGUI m){
            this.m= m;
            setPreferredSize(new Dimension(tileWidth*width, tileHeight*height));
        }
        
        /* Paint the game board. */
        public void paint(Graphics g){
            //paint the background
            g.setColor(Color.BLACK);
            g.fillRect(0, 0, m.width*m.tileWidth, m.height*m.tileHeight);
            
            // Paint each element of the game board
            for (int i= 0; i < m.width; i= i+1){
                for (int j= 0; j < m.height; j= j+1){
                    // tile (i, j) is in pixels (h..h1-1, v..v1-1)
                    int h= i * m.tileWidth;
                    int h1= (i+1) * m.tileWidth;
                    int v= j * m.tileHeight;
                    int v1= (j+1) * m.tileHeight;
                    
                    if (m.board[i][j] != null) {
                        g.setColor(m.board[i][j].getColor());
                        
                        if (m.board[i][j].getType() == Piece.BLOCK){
                            // Tile is a block; fill it with a square.
                            g.fillRect(h+1, v, m.tileWidth-2, m.tileHeight-2);
                            
                        } else if (m.board[i][j].getType() == Piece.JMAN){
                            // Fill J*Man's square with J*Man's Asterix Icon.
                            g.drawLine(h+3, v+2, h1-3, v1-2);
                            g.drawLine(h+3, v1-2, h1-3, v+2);
                            g.drawLine(h+1, v+m.tileHeight/2, h1-1, v+m.tileHeight/2);
                            g.drawLine(h+m.tileWidth/2, v+1, h+m.tileWidth/2, v1-1);
                            
                        } else if (m.board[i][j].getType() == Piece.WALKER){
                            //Tile is a walker, fill it with an appropriate colored triangle.
                            g.fillPolygon(new int[]{h+1, h1-1, h+m.tileWidth/2},
                                          new int[]{v1-2, v1-2, v}, 3);
                            
                        } else if (m.board[i][j].getType() == Piece.PILLAR){
                            // Tile is a pillar, fill it with an appropriate colored disk.
                            g.fillOval(h+1, v, m.tileWidth-2, m.tileHeight-2);
                        }
                    }
                }
            }
        }
    }// end of inner class JManPanel
}