using System;
using System.Collections.Generic;
using System.Text;

using System.Drawing;
using System.IO;
using System.Net.Sockets;

/* This class actually talks to the Aibo's camera interface.
 * Things that could be improved:
 *   - Not making a new TCP connection for every frame
 *   - Perhaps some more comments
 * 
 * --cgd
 */

namespace Cornell.Cs100r.Robotics.DssAiboCamera
{
    public class FrameGrabber
    {
        public static readonly int AIBO_CAMERA_PORT = 6002;

        static void toRGB(byte Y, byte Cb, byte Cr, out byte R, out byte G, out byte B)
        {
            /* This is taken from:
             * http://tinyurl.com/elos2
             *
             * I tried about five different formulations of converting 
             * YCbCr to RGB, and this one worked the best.  It was also 
             * the fastest.
             * --cgd */
            int C = Y - 16;
            int D = Cb - 128;
            int E = Cr - 128;

            R = clip(((298 * C + 409 * E + 128) >> 8));
            G = clip(((298 * C - 100 * D - 208 * E + 128) >> 8));
            B = clip(((298 * C + 516 * D + 128) >> 8));
        }

        static Color toRGB(byte Y, byte Cb, byte Cr)
        {
            byte R, G, B;
            toRGB(Y, Cb, Cr, out R, out G, out B);
            return Color.FromArgb(R, G, B);
        }
        static byte clip(int a)
        {
            if (a < 0) return 0;
            if (a > 255) return 255;
            return (byte)a;
        }

        static Bitmap grabFrameFromStream(Stream s)
        {
#if MEASURE
            Win32.HiPerfTimer t = new Win32.HiPerfTimer();
            t.Start();
#endif
            /* When I was reverse-engineering this I did a lot of work 
             * against a MemoryStream of a packet dump, so this is here.
             * I see no reason to take it out.
             * --cgd */
            if (s is NetworkStream)
                writeBytes((NetworkStream)s, BitConverter.GetBytes((UInt32)0x03));

            UInt32 totalSize = readUInt32(s);
            UInt32 numData = readUInt32(s);

            if (numData != 1) { return null; }

            UInt32 bufSize = readUInt32(s);

            UInt32[] header = new UInt32[8];

            for (int i = 0; i < header.Length; i++)
            {
                header[i] = readUInt32(s);
            }

            UInt32 bytesPerElement = header[2];

            if (bytesPerElement != 3) { return null; }

            UInt32 m = header[3];
            UInt32 n = header[4];

            UInt32 nlength = bytesPerElement * m * n;

            byte[] values = new byte[nlength];
            readn(s, values, 0, (int)nlength);

#if MEASURE
            t.Stop();
            System.Console.Out.WriteLine("Received all data after connect in {0} secs", t.Duration);
            
            t.Start();
#endif
            Bitmap img = new Bitmap((int)n, (int)m, System.Drawing.Imaging.PixelFormat.Format24bppRgb);

            /* This scary stuff is needed to reduce the processing time
             * from 3.27 seconds to about 0.01 to 0.003 seconds (big difference!).
             * Adapted from:
             * http://msdn2.microsoft.com/en-us/library/system.drawing.imaging.bitmapdata.scan0.aspx 
               --cgd */

            Rectangle rect = new Rectangle(0, 0, img.Width, img.Height);
            System.Drawing.Imaging.BitmapData bmpData = img.LockBits(rect,
                System.Drawing.Imaging.ImageLockMode.ReadWrite,
                img.PixelFormat);

            // Start of image data in memory
            IntPtr ptr = bmpData.Scan0;

            /* We specified 24bpp (3 bytes/pixel).  So we can just
             * reuse nlength as our # of bytes to allocate here */
            int nbytes = (int)nlength;

            /* Read YCbCr from the array and convert to RGB in-place.
             * NB: We are assuming here that the width of the image is divisible
             * by 4.  If this is not the case, the stride of the Bitmap object
             * (the number of bytes per scan line) will NOT equal the width,
             * because they pad it out to be a multiple of 4.  This assumption
             * is true for the Aibo's camera.  --cgd */
            int addr = 0;
            for (int i = 0; i < m; i++)
            {
                for (int j = 0; j < n; j++)
                {
                    byte Y = values[addr];
                    byte Cb = values[addr + 1];
                    byte Cr = values[addr + 2];
                    byte R, G, B;

                    toRGB(Y, Cb, Cr, out R, out G, out B);

                    // The data are arranged BGRBGRBGRBGR (I think because of little-endianness)
                    values[addr++] = B;
                    values[addr++] = G;
                    values[addr++] = R;
                }
            }

            // Copy the data back
            System.Runtime.InteropServices.Marshal.Copy(values, 0, ptr, nbytes);

            // And unlock the data from system memory
            img.UnlockBits(bmpData);

#if MEASURE
            t.Stop();
            System.Console.Out.WriteLine("Converted & loaded data in {0} secs", t.Duration);
#endif
            /* This probably isn't needed (I've never actually seen it be needed
             * in any packet trace) but, well, the Matlab implementation does it,
             * so I will too. --cgd */
            int ndiscard = (int)bufSize - 32 - nbytes;
            if (ndiscard > 0)
            {
                byte[] discard = new byte[ndiscard];
                s.Read(discard, 0, ndiscard);
            }

            s.Close();

            return img;
        }

        public static Bitmap getFrame(string aiboAddress)
        {

            TcpClient sock;
#if MEASURE
            Win32.HiPerfTimer t = new Win32.HiPerfTimer();
            t.Start();
#endif
            sock = new TcpClient(aiboAddress, AIBO_CAMERA_PORT);
            NetworkStream s = sock.GetStream();

#if MEASURE
            t.Stop();
            System.Console.Out.WriteLine("Took {0} secs to connect", t.Duration);


            t.Start();
#endif
            Bitmap img = grabFrameFromStream(s);
#if MEASURE
            t.Stop();
            System.Console.Out.WriteLine("Took {0} secs to grab frame", t.Duration);
#endif
            return img;
        }

        /* Some helper functions because I'm lazy */
        static void writeBytes(NetworkStream s, byte[] arr)
        {
            s.Write(arr, 0, arr.Length);
        }

        /* This assumes data is transmitted little-endian
         * (that's how UPenn rolls) */
        static UInt32 readUInt32(Stream s)
        {
            byte[] a = new byte[4];
            readn(s, a, 0, 4);
            return BitConverter.ToUInt32(a, 0);
        }

        /* It kind of annoys me that I had to implement this */
        static int readn(Stream s, byte[] buffer, int offset, int n)
        {
            int c = 0;
            while (c < n)
            {
                c += s.Read(buffer, offset + c, n - c);
            }
            return c;
        }


    }

}
