classdef Game < handle
    % An escape-the-cave game.

    properties (Access=private)
        player  % Player character
        monster  % Monster character
        map  % Map of rooms

        title  % Handle to graphics title
    end

    methods

        function obj = Game(n)
            % Construct a new Game object with an n-by-n board.

            % Set up graphics
            close all

            % DEMO: Set object properties with dot notation
            f = figure();
            % Note: For compatibility with various operating systems, must
            % set window position as well as size.
            f.Position = [100, 100, 100*n, 100*n + 15];
            f.Color = [0.7 0.4 0.1];

            axis equal off
            hold on

            % DEMO: Control text formatting with LaTeX markup
            text(1, n+1.2, ...
                '\color[rgb]{0.9,0.8,0.4}\fontsize{16}\bf The cave!')

            % DEMO: Initialize object properties with key-value function arguments
            obj.title = title('', 'Color', [1, 1, 1], 'FontSize', 13);

            % TODO: Generate map
            % 1. Initialize `map` property
            % 2. Construct rooms on n-by-n grid and add to map
            %    A. Each room has a 10% chance of being a trap room and a
            %       20% chance of being a poison room
            %    B. Start and end rooms must not be hazardous


            % Customize start and exit rooms
            start = obj.map.getRoom([1,1]);
            start.addText('Start!')
            exit = obj.map.getRoom([n,n]);
            exit.addText('Exit here!')
            exit.setColor([1, 1, 0])

            % Create characters
            obj.player = Player(start, obj.map);
            obj.monster = Monster(exit, obj.map);
        end

        function run(self)
            % Run the game, taking turns between characters until the
            % player either escapes, encounters the monster, or runs out of
            % health.

            alive = true;  % Whether player is still alive
            c = self.player;  % Active character

            % Loop until game-over condition
            while alive && any(self.player.getPos() ~= self.map.getExitPos())
                % Show status in title
                self.showMessages()

                % Active character takes action and responds to environment
                c.sense(self)
                c.move()
                c.update()

                % Check if player has been eaten or mortally wounded
                if all(self.monster.getPos() == self.player.getPos()) || ...
                        self.player.getHealth() <= 0
                    alive = false;
                end

                % Determine which character gets to move next
                c = self.nextCharacter(c);
            end

            % Show end-game messages in title
            if alive
                self.title.String = 'You reached the exit! Yay, you win!';
            elseif all(self.monster.getPos() == self.player.getPos())
                self.title.String = 'OH NO! You were caught by the monster!';
            else
                self.title.String = 'Game Over! You lost all your health.';
            end
        end

        function p = getPlayerPos(self)
            % Query the current position ([x,y]) of the player character.
            p = self.player.getPos();
        end

    end

    methods (Access=private)

        function c = nextCharacter(self, c)
            % Given the current active character (input `c`), return the
            % character who should move next.

            if ~c.hasMovesRemaining()
                % Swap active character
                if c == self.monster
                    c = self.player;
                else
                    c = self.monster;
                end

                % Reset number of moves at start of turn
                c.startTurn();
            end
            % Otherwise, active character can still move, so don't change `c`.
        end

        function showMessages(self)
            % Set title text to show current status of player.

            s = sprintf('Health: %d', self.player.getHealth());
            if self.player.isPoisoned
                s = [s, ' [poisoned]'];
            end
            s = [s, sprintf('; %d move(s) left', self.player.getMovesRemaining)];
            self.title.String = s;
        end

    end

end
