classdef Character < handle
    % Base class of characters (both player and non-player) in cave game.

    properties (Access=protected)
        room  % Room this character is currently in
        map  % Map of cave character is constrained to
        movesRemaining  % Number of moves character may still make this turn
    end

    % Graphics properties
    properties (Access=private)
        marker  % Graphics object for character marker
        label  % Graphics object for character marker label
    end

    methods
        function obj = Character(start, map, color, label)
            % Construct a character in Room `start` within Map `map` whose
            % marker has the color `color` and label `label` (one
            % character).

            obj.room = start;
            obj.map = map;

            % Initialize number of remaining moves
            obj.startTurn()

            % Set up graphics
            p = obj.room.getPos();
            obj.marker = fill([p(1)+0.05, p(1)+0.3, p(1)+0.3, p(1)+0.05], ...
                [p(2)+0.05, p(2)+0.05, p(2)+0.3, p(2)+0.3], ...
                color, 'LineStyle', 'none');
            obj.label = text(p(1)+0.12, p(2)+0.18, label);
            obj.label.Color = [1,1,1];
        end

        function p = getPos(self)
            % Return position ([x,y]) of room this character is in.
            p = self.room.getPos();
        end

        function h = hasMovesRemaining(self)
            % Return true of this character may still move this turn.
            h = self.getMovesRemaining() > 0;
        end

        function m = getMovesRemaining(self)
            % Return how many times the character may still move this turn.
            m = self.movesRemaining;
        end

        function startTurn(self)
            % Reset number of remaining moves for the start of a new turn.
            self.movesRemaining = 1;
        end

        function endTurn(self)
            % Set number of remaining moves to zero.
            self.movesRemaining = 0;
        end

        function sense(self, game)
            % Observe state of Game `game` (opportunity to get locations of
            % other characters).

            % Base class does nothing
        end

        function move(self)
            % Move to a new room in direction specified by subclass.

            dir = self.nextDir();  % Subclass must implement `getDir()`
            self.room = self.map.getRoom(self.room.getPos() + dir);
            self.movesRemaining = self.movesRemaining - 1;

            % Update graphics
            p = self.room.getPos();
            self.marker.Vertices = [p(1)+0.05, p(1)+0.3, p(1)+0.3, p(1)+0.05; ...
                p(2)+0.05, p(2)+0.05, p(2)+0.3, p(2)+0.3]';
            self.label.Position = [p(1)+0.12, p(2)+0.18];
        end

        function update(self)
            % Update character state based on current environment.

            % Base class does nothing
        end

    end

end
