classdef Game < handle
    
    % ---- Do not modify this class ---- %

    properties
        roomArr         % Square matrix of Room objects
        player          % Reference to a Player object
        monster         % Reference to a Monster object
        exitRoom        % Reference to Room object
        numRooms        % Number of rooms (must be square)
        type            % If 0, monster moves to attack. If 1, monster 
                        % moves to protect. Else, monster randomly chooses 
                        % between attacking and protecting
    end %public properties
    
    methods
        
        function g = Game(numRooms, type, hazardChance)
        % Game constructor. hazardChance is the probability that a
        % room will be hazardous.
            if (nargin <= 2)
                hazardChance=0;
            end
            if (nargin >= 1)
                g.numRooms=numRooms;
                gridLength = floor(sqrt(numRooms));
                g.roomArr = Room();
                i = 0;
                for x = 1:gridLength
                    for y = 1:gridLength
                        i = i+1;
                        hazard=0;
                        if rand<hazardChance
                            hazard=round(rand)+1;
                        end
                        g.roomArr(x,y) = Room(x,y,0,i,0,0,hazard);
                    end
                end
                % The player always starts at the bottom-left room
                g.player = Player(g.roomArr(1,1),...
                                  100,Room.hazardAmount/2,.7);
                g.roomArr(1,1).playerVisited=1;
                
                % The exit room is always the top-right room.
                g.exitRoom=g.roomArr(gridLength, gridLength);
                g.exitRoom.exit=1;
                
                % The exit room and player's start room cannot be hazardous
                g.exitRoom.hazardID=0;
                g.roomArr(1,1).hazardID=0;
                
                % The monster always begins at the exit room
                g.monster = Monster(g.exitRoom);
                g.type=0;
            end
            if (nargin >= 2)
                g.type=type;
            end
        end
        
        function [x,y]= getMonstLoc(self)
            i=0;
            x=-1;
            while x == -1
                i=i+1;
                r=self.roomArr(i);
                if r.monsterInRoom
                    [x, y]=r.getLoc();
                end
            end
        end
        
        function [x,y]= getPlayerLoc(self)
            i=0;
            x=-1;
            while x == -1
                i=i+1;
                r=self.roomArr(i);
                if r.playerInRoom
                    [x, y]=r.getLoc();
                end
            end
        end
        
        function dist = getMonstDist(self)
            i=1;
            mx=-1;
            px=-1;
            while mx == -1 || px == -1
                r=self.roomArr(i);
                if r.playerInRoom
                    [px, py]=r.getLoc();
                end
                if r.monsterInRoom
                    [mx, my]=r.getLoc();
                end
                i=i+1;
            end
            dist = (sqrt((px-mx)^2+(py-my)^2));
        end
        
        function figHandle=draw(self, figHandle)
        % Draws the initial game setup
            if isempty(figHandle)
                figHandle = figure('Color',[0.7 0.4 0.1],'Position', ...
                                   [100,100,100*sqrt(self.numRooms), ...
                                    100*sqrt(self.numRooms)+15]);
            end
            hold on
            axis([1 sqrt(self.numRooms)+1.5 1 sqrt(self.numRooms)+1.5])
            axis off
            text(1,sqrt(self.numRooms)+1.2,...
                '\color[rgb]{0.9,0.8,0.4}\fontsize{16}\bf The cave!')
            for i=1:sqrt(self.numRooms)
                for j=1:sqrt(self.numRooms)
                    self.roomArr(i,j).draw();
                end
            end
            self.player.draw();
            self.monster.draw();
        end
        
        function [wasPoisoned, fid] = updateGraphics(self, wasPoisoned, fid)
            % Get new position for player and redraw it
            i = 1;
            won = 0;
            while(i<=2 && ~won)
                fid=self.draw(fid);
                [newX,newY] = ginput(1);
                newX = floor(newX);
                newY = floor(newY);
                oldX = self.player.room.xCoord;
                oldY = self.player.room.yCoord;
                gridLength= length(self.roomArr);
                dx = 0;
                dy = 0;
                if newX>=1 && newX<=gridLength && ...
                   newY>=1 && newY<=gridLength
                    if (newX > oldX && newY == oldY)       % move east
                        dx = 1;
                    elseif (newX > oldX && newY > oldY)    % move northeast
                        dx = 1;
                        dy = 1;
                    elseif (newX > oldX && newY < oldY)    % move southeast
                        dx = 1;
                        dy = -1;
                    elseif (newX < oldX && newY == oldY)   % move west
                        dx = -1;
                    elseif (newX < oldX && newY > oldY)    % move northwest
                        dx = -1;
                        dy = 1;
                    elseif (newX < oldX && newY < oldY)    % move southwest
                        dx = -1;
                        dy = -1;
                    elseif (newX == oldX && newY > oldY)   % move north
                        dy = 1;
                    elseif (newX == oldX && newY < oldY)   % move south
                        dy = -1;
                    end
                end
                self.player.removeDrawing();
                self.player.move(self.roomArr,dx,dy);
                self.player.room.draw();
                self.player.draw();
                
                out=sprintf('Monster is %.2f spaces away!\n', ...
                            self.getMonstDist());
                out=[out sprintf('You have %.0f health points\n', ...
                                 self.player.getHealth)];
                if self.player.room.hazardID==1
                    out=[out sprintf(...
                         'You were hit by a trap! (-20 health points)\n')];
                end
                if self.player.checkPoison
                    out=[out sprintf(...
                         'You are poisoned! (-10 health points per move)')];
                    wasPoisoned=1;
                end
                if ~self.player.checkPoison && wasPoisoned
                    out=[out sprintf('You recovered from poison!')];
                    wasPoisoned=0;
                end
                title(out,'Color','white','FontSize',13);
                
                if self.player.room.hazardID > 0
                    sprintf('Room #%d is hazardous!', ...
                            self.player.room.getID());
                end
                if self.player.room.exit == 1
                    title('You reached the exit! Yay, you win!');
                    won= 1;
                end
                i = i+1;
            end
            
            if won || self.player.room.monsterInRoom
                return
            else
                % Update location of monster and draw it
                self.monster.removeDrawing();
                if self.type==1
                    self.monster.moveToAttack(self);
                elseif self.type==2
                    self.monster.moveToProtect(self);
                else
                    if rand>0.5
                        self.monster.moveToAttack(self);
                    else
                        self.monster.moveToProtect(self);
                    end
                end
                self.monster.draw();
            end
        end
        
        function run(self)
            wP=0;
            fID=[];
            while(self.player.room.exit == 0 && ...
                    self.player.room.getID~=self.monster.room.getID ...
                    && self.player.getHealth>0)
                [wP, fID]=self.updateGraphics(wP, fID);
            end
            [~]=self.draw(fID);
            if self.player.room.exit == 1
                title('You reached the exit! Yay, you win!', ...
                      'Color','white','FontSize',13);
            elseif self.player.room.getID==self.monster.room.getID
                title('OH NO! You were caught by the monster!', ...
                      'Color','white','FontSize',13);
            elseif self.player.getHealth<=0
                title('You have been defeated by... The Cave.', ...
                      'Color','white','FontSize',13);
            end
        end
        
    end % public methods
    
end % class Game