import java.awt.*;
import java.awt.image.BufferedImage;
import javax.imageio.ImageIO;
import java.io.File;
import javax.swing.*;

/* Defines a stained glass image, i.e., stained glass effect applied on an
  image.
  The stained glass effect comprises nr rows by nc columns of tiles.  Using
  a regular grid, each tile is rectangular and has a color obtained from 
  averaging the pixels in the tile.  In drawing the tile the interior grid
  points are perturbed so that the tiles appear as irregular quadrilaterals.
  */

public class StainedGlassImage extends JPanel{
  private int[][][] imData;  // array of uint8s (color values)
  private int nr;            // number of rows of tiles
  private int nc;            // number of columns of tiles
  private Tile[][] tileSet;  // 2-d cell array of Tile references
  private Vertex[][] grid;   // 2-d cell array of Vertex references
  private double frac = 0.0; // fraction for perturbing interior grid points
  
  // Constructor: Create StainedGlassImage object from jpeg file named
  // by fname, using nr rows and nc columns of tiles
  public StainedGlassImage(String fname, int nr, int nc, double noiseFrac){
    try{
      this.imData = imread(fname);
    } catch (Exception e) {
      System.out.println(e.getMessage());
    }
    this.nr = nr;
    this.nc = nc;
    this.frac = noiseFrac;
    this.tileSet = new Tile[nr][nc];
    this.grid = new Vertex[nr+1][nc+1];
    applyEffect();
  }
  
  private void applyEffect(){
    int nrIm = imData.length;
    int ncIm = imData[0].length;
    int np = imData[0][0].length;
    
    // Beginning pixel row and colum indices of the tiles 
    int[] rstarts = new int[nr+1];
    int[] cstarts = new int[nc+1];
    // rstarts(r) is the starting pixel row index in the rth tile on a row, r=1:nr
    for (int i=0; i<nr; i++){
      rstarts[i] = (int)((double)i/nr*nrIm);
    }
    rstarts[nr] = nrIm;
    // cstarts(c) is the starting pixel column index in cth tile on a column, c=1:nc
    for (int j=0; j<nc; j++){
      cstarts[j] = (int)((double)j/nc*ncIm);
    }
    cstarts[nc] = ncIm;
    
    int minDim = Math.min(nr,nc);
    double maxNoise = frac*minDim;
    
    // Create the grid (a set of Vertexs)
    for (int r=0; r<nr+1; r++){
      for (int c=0; c<nc+1; c++){
        boolean isEdge = (r==0)||(r==nr)||(c==0)||(c==nc);
        grid[r][c] = new Vertex(rstarts[r], cstarts[c], isEdge, maxNoise);
      }
    }
    
    // Create the tile set (a set of Tiles)
    for (int r=0; r<nr; r++){
      for (int c=0; c<nc; c++){
        double[] colr = {0.0, 0.0, 0.0};
        for (int p=0; p<np; p++){
          // Get color (average data) for the tile at row r column c
          double sum = 0.0;
          int sumnum = 0;
          for (int i=rstarts[r]; i<rstarts[r+1]; i++){
            for (int j=cstarts[c]; j<cstarts[c+1]; j++){
              sum = sum + (double)imData[i][j][p]/255;
              sumnum = sumnum + 1;
            }
          }
          colr[p] = sum/sumnum;
        }
        // The tile at row r column c has these 4 vertices:
        //   (r,c), (r,c+1), (r+1,c+1), and (r+1,c)
        Vertex[] vertices = {grid[r][c], grid[r][c+1], grid[r+1][c+1], grid[r+1][c]};
        // Make the Tile at row r column c
        tileSet[r][c] = new Tile(colr, vertices);
      }
    }
  }
  
  public void paint(Graphics g) {
    for (int r=0; r<nr; r++){
      for (int c=0; c<nc; c++){
        tileSet[r][c].draw(g);
      }
    }
  }
  
  // java implementation of imread
  public int[][][] imread(String fname) throws Exception{
    BufferedImage image = ImageIO.read(new File(fname));
    int w = image.getWidth();
    int h = image.getHeight();
    int[][][] imData = new int[w][h][3];

    for (int r=0; r<h; r++) {
      for (int c=0; c<w; c++) {
        int rgb = image.getRGB(c,r);
        Color clr = new Color(rgb);
        imData[c][r][0] = clr.getRed();
        imData[c][r][1] = clr.getGreen();
        imData[c][r][2] = clr.getBlue();
      }
    }
    return imData;
  }
  
  public static void main(String[] args) {
    JFrame frame = new JFrame();
    frame.getContentPane().add(new StainedGlassImage("./usainBolt.jpg",30,25,0.2));
    
    frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    frame.setSize(398,429);
    frame.setVisible(true);
    frame.setTitle("StainedGlassImage");
  }
}