# controller.py
# Walker M. White (wmw2)
# October 19, 2012
"""
Controller class/module for temperature converter

This class represents the CONTROLLER.  The controller is responsible for sending messages
between the view and the model (e.g it calls methods in those objects and sends the
return values back and forth) . Therefore, it needs to have attributes referencing the
model and the view so that it can access each one of them. In most cases (including this
one), the controller is also responsible for starting the program. That is what the run()
method does.

This class has some features, such as callback functions, that we will not introduce in
this course.

Author: Walker M. White (wmw2)
Date:   October 20, 2020
"""

from kivy.app import App
from kivy.config import Config

# Coordinate the other classes
import view
import model

class ConverterApp(App):
    """
    Class represents a controller for a converter
    application.

    The controller contains the code for initializing
    the application. It also contains the code for
    responding communicating the information typed to
    the model (e.g. it calls the getters and setters
    in the model when the user types anything).
    """
    # INSTANCE ATTRIBUTES
    # Attribute _model: The converter model
    # Invariant: _model is Temperature object
    #
    # Attribute _view: The converter view
    # Invariant: _view is a TemperaturePanel object

    def build(self):
        """
        Reads the kivy file and performs layout
        """
        Config.set('graphics', 'position', 'custom')
        Config.set('graphics', 'left', '150')
        Config.set('graphics', 'top', '150')
        Config.set('graphics', 'width', '500')
        Config.set('graphics', 'height', '200')

        # Initialize the model and the view
        self._view  = view.TemperaturePanel()
        self._model = model.Temperature()

        # Initialize the text field values.
        self.updateView()

        # Add callbacks to react to changes in input.
        # These are the methods called when the user clicks on the text
        self._view.linkFahrenheit(self.updateFahrenheit)
        self._view.linkCentigrade(self.updateCentigrade)

        return self._view # Kivy needs view to start

    def updateView(self):
        """
        Updates the view to be consistent with the model
        """
        self._view.setFahrenheitText(str(round(self._model.getFahrenheit(),4)))
        self._view.setCentigradeText(str(round(self._model.getCentigrade(),4)))


    def updateFahrenheit(self,field):
        """
        Reacts to a change in the fahrenheit text field

        Parameter field: The text field associated with fahrenheit
        Precondition: field is a Kivy TextInput field
        """
        result = field.text
        value = 0.0
        try:
            # This might fail
            value = float(result)
        except:
            field.text = str(value)
        self._model.setFahrenheit(value)
        self.updateView()

    def updateCentigrade(self,field):
        """
        Reacts to a change in the centigrade text field

        Parameter field: The text field associated with centigrade
        Precondition: field is a Kivy TextInput field
        """
        result = field.text
        value = 0.0
        try:
            # This might fail
            value = float(result)
        except:
            field.text = str(value)
        self._model.setCentigrade(value)
        self.updateView()
