"""
Model class/module for temperature converter.

This is the MODEL that holds our temperature values and
converts between them for us. We separate this from the
view so that we can restore the old state whenever a user
enters bad data. If the view took care of everything, we
would not be able to undo bad input.

At any given time, this model should be consistent. That
is, the fahrenheit and the centigrade value should never
get out of synch. Therefore, whenever the fahrenheit value
is changed, the centigrade value must be changed immediately
as well.  This is a prime example of why you should NEVER
make attributes public, but use getters and setters instead.

In order to ensure consistency, we will only have one
attribute: the fahrenheit attribute. The centigrade
attribute is computed on the fly whenever we need it
(e.g. when the getter is called).

Note that the model NEVER "talks" to the controller or
the view (e.g  it never calls a method in those objects).
The model is an entirely passive object; the controller
does all of the talking by calling its getters/setter.
This allows for maximum code reuse.

Author: Walker M. White (wmw2)
Date:   October 20, 2020
"""


class Temperature(object):
    """
    A class representing a temperature value.

    The class has two attributes for fahrenheit and
    centigrade (accessible through the getters and setters)
    which it keeps in sync with each other.  That is
    fahrenheit will always be equal to 9*centigrade/5.0+32.
    """
    # INSTANCE ATTRIBUTES
    # Attribute _fahrenheit: The temperature in fahrenheit
    # Invariant: _fahrenheit is a float

    def getFahrenheit(self):
        """
        Returns temperature value in farenheit

        This value is guaranteed to be equal to 9*centigrade/5+32
        """
        return self._fahrenheit

    def setFahrenheit(self,value):
        """
        Sets temperature value in fahrenheit

        Parameter value: The new temperature in fahrenheit
        Precondition: value is a number (int or float)
        """
        assert type(value) == int or type(value) == float, repr(value)+' is not a number'
        self._fahrenheit = float(value) # Enforce the invariant

    def getCentigrade(self):
        """
        Returns temperature value in centigrade

        This value is guaranteed to be equal to 5*(farenheit-32)/9
        """
        return 5*(self._fahrenheit-32)/9.0

    def setCentigrade(self,value):
        """
        Sets temperature value in centigrade

        Parameter value: The new temperature in centigrade
        Precondition: value is a number (int or float)
        """
        assert type(value) == int or type(value) == float, repr(value)+' is not a number'
        self._fahrenheit = 9*float(value)/5.0+32 # Enforce the invariant

    def __init__(self, fahrenheit=None,centigrade=None):
        """
        Initializes a temperature with the given farenheit
        or centigrade value.

        Parameter fahrenheit: The temperature in fahrenheit
        Precondition: fahrenheit is a number (int or float) or None

        Parameter centigrade: The temperature in centigrade
        Precondition: centigrade is a number (int or float) or None

        At least one of centigrade or fahrenheit must be None (to prevent conflicts).
        If both are None, the temperature is set to 0 centigrade.
        """
        assert fahrenheit is None or centigrade is None, 'You can only have one temparature parameter'
        if not fahrenheit is None:
            self.setFahrenheit(fahrenheit)
        elif not centigrade is None:
            self.setCentigrade(centigrade)
        else:
            self.setCentigrade(0)

    def __repr__(self):
        """
        Returns the unambiguous tring representation of this
        temperature
        """
        return str(self.__class__) + str(self)

    def __str__(self):
        """
        Returns a string representation of this temperature
        """
        return str(self.fahrenheit) + ' F (' + str(self.centigrade) + ' C)'

    def __eq__(self,other):
        """
        Returns True if other is an equal temperature,
        False otherwise.

        Parameter other: The object to compare
        Precondition: NONE (other can be anything)
        """
        if not instanceof(other,Temperature):
            return false
        # Compare floats by checking if the difference is small.
        return abs(other.fahrenheit-self.fahrenheit) < 1e-8
