"""
A module with a simple Fraction class and its subclass.

This module shows off why we want to use isinstance.

Author: Walker M. White (wmw2)
Date:   November 10, 2022
"""
import math


def gcd(a,b):
    """
    Returns: Greatest common divisor of x and y.

    Precondition: x and y are integers.
    """
    assert type(a) == int,repr(x)+' is not an int'
    assert type(b) == int,repr(y)+' is not an int'
    while b != 0:
       t = b
       b = a % b
       a = t
    return a


class Fraction(object):
    """
    A class to represent a fraction n/d
    """
    # INSTANCE ATTRIBUTES
    # Attribute _numerator: The fraction numerator
    # Invariant: _numerator is an int
    #
    # Attribute _denominator: The fraction denominator
    # Invariant: _denominator is an int > 0

    # GETTER AND SETTERS
    def getNumerator(self):
        """
        Returns the fraction numerator.

        The numerator is an int.
        """
        return self._numerator # returns the attribute

    def setNumerator(self,value):
        """
        Sets the numerator to value.

        Parameter value: the new numerator
        Precondition: value is an int
        """
        # enforce invariant
        assert isinstance(value, int), repr(value)+' is not an int'
        # assign to attribute
        self._numerator = value

    def getDenominator(self):
        """
        Returns the fraction denominator.

        The denominator is an int > 0.
        """
        return self._denominator # returns the attribute

    def setDenominator(self,value):
        """
        Sets the numerator to value.

        Parameter value: the new denominator
        Precondition: value is an int > 0
        """
        # enforce invariant
        assert isinstance(value, int), repr(value)+' is not an int'
        assert value > 0, repr(value)+' is not positive'
        # assign to attribute
        self._denominator = value

    # INITIALIZER
    def __init__(self,n=0,d=1):
        """
        Initializes a new Fraction n/d

        Parameter n: the numerator (default is 0)
        Precondition: n is an int (or optional)

        Parameter d: the denominator (default is 1)
        Precondition: d is an int > 0 (or optional)
        """
        # No need for asserts; setters handle everything
        self.setNumerator(n)
        self.setDenominator(d)

    def __str__(self):
        """
        Returns this Fraction as a string 'n/d'
        """
        return repr(self._numerator)+'/'+repr(self._denominator)

    def __repr__(self):
        """
        Returns the unambiguous representation of Fraction
        """
        return str(self.__class__)+'['+str(self)+']'

    # MATH METHODS
    def __mul__(self,other):
        """
        Returns the product of self and other as a new Fraction

        This method does not modify the contents of self or other

        Parameter other: the value to multiply on the right
        Precondition: other is a Fraction or an int
        """
        #assert type(other) == Fraction or type(other) == int
        assert isinstance(other,Fraction) or isinstance(other,int), repr(other)+' is not a valid operand'
        if type(other) == int:
            return self._multiplyInt(other)
        return self._multiplyFraction(other)

    # Private helper to multiply fractions
    def _multiplyFraction(self,other):
        """
        Returns the product of self and other as a new Fraction

        This method does not modify the contents of self or other

        Parameter other: the fraction to multiply on the right
        Precondition: other is a Fraction
        """
        # No need to enforce preconditions on a hidden method
        top = self.getNumerator()*other.getNumerator()
        bot = self.getDenominator()*other.getDenominator()
        return Fraction(top,bot)

    # Private helper to multiply ints
    def _multiplyInt(self,x):
        """
        Returns the product of self and other as a new Fraction

        This method does not modify the contents of self or other

        Parameter other: the value to multiply on the right
        Precondition: other is a int
        """
        # No need to enforce preconditions on a hidden method
        top = self.getNumerator()*x
        bot = self.getDenominator()
        return Fraction(top,bot)

    def __add__(self,other):
        """
        Returns the sum of self and other as a new Fraction

        This method does not modify the contents of self or other

        Parameter other: the value to add on the right
        Precondition: other is a Fraction or an int
        """
        assert isinstance(other,Fraction) or isinstance(other,int), repr(other)+' is not a valid operand'
        if type(other) == int:
            return self._addInt(other)
        return self._addFraction(other)

    # Private helper to multiply fractions
    def _addFraction(self,other):
        """
        Returns the sum of self and other as a new Fraction

        This method does not modify the contents of self or other

        Parameter other: the fraction to add on the right
        Precondition: other is a Fraction
        """
        # No need to enforce preconditions on a hidden method
        bot = self.getDenominator()*other.getDenominator()
        top = (self.getNumerator()*other.getDenominator()+
               self.getDenominator()*other.getNumerator())
        return Fraction(top,bot)

    # Private helper to multiply ints
    def _addInt(self,x):
        """
        Returns the sum of self and other as a new Fraction

        This method does not modify the contents of self or other

        Parameter other: the value to add on the right
        Precondition: other is an int
        """
        # No need to enforce preconditions on a hidden method
        bot = self.getDenominator()
        top = (self.getNumerator()+self.getDenominator()*x)
        return Fraction(top,bot)

    # COMPARISONS
    def __eq__(self,other):
        """
        Returns True if self, other are equal Fractions.

        It returns False if they are not equal, or other is not a Fraction

        Parameter other: value to compare to this fraction
        Precondition: NONE
        """
        if not isinstance(other, Fraction) and not isinstance(other, int):
            return False

        if isinstance(other, int):
            return self.getNumerator() == other*self.getDenominator()

        # Cross multiply
        left = self.getNumerator()*other.getDenominator()
        rght = self.getDenominator()*other.getNumerator()
        return left == rght

    def __lt__(self,other):
        """
        Returns True if self < other, False otherwise

        This method is used to implement all strict comparison operations.  Both < and >
        are determined automatically from this method.

        Parameter other: value to compare to this fraction
        Precondition: other is a Fraction
        """
        assert isinstance(other, Fraction) or isinstance(other, int), repr(other)+' is not a valid operand'

        if isinstance(other, int):
            return self.getNumerator() < other*self.getDenominator()

        # Cross multiply
        left = self.getNumerator()*other.getDenominator()
        rght = self.getDenominator()*other.getNumerator()
        return left < rght

    def __le__(self,other):
        """
        Returns True if self < other, False otherwise

        This method is used to implement all inclusive comparison operations.  Both <=
        and >= are determined automatically from this method.

        Parameter other: value to compare to this fraction
        Precondition: other is a Fraction
        """
        assert isinstance(other, Fraction) or isinstance(other, int), repr(other)+' is not a valid operand'

        if isinstance(other, int):
            return self.getNumerator() <= other*self.getDenominator()

        # Cross multiply
        left = self.getNumerator()*other.getDenominator()
        rght = self.getDenominator()*other.getNumerator()
        return left <= rght

    # OTHER METHODS
    def reduce(self):
        """
        Normalizes this fraction into simplest form.

        Normalization ensures that the numerator and denominator have no
        common divisors.
        """
        g = gcd(self.getNumerator(),self.getDenominator())
        self.setNumerator(self.getNumerator()//g)
        self.setDenominator(self.getDenominator()//g)


class FractionalLength(Fraction):
    """
    A class to represent fractions with a unit of measurement
    """
    # Attribute _units: The unit of measurement
    # Invariant: _unit is one of 'in', 'ft', 'yd'
    
    def getUnits(self):
        """
        Returns the unit of measurement for this length
        """
        return self._units
    
    def setUnits(self,value):
        """
        Sets the unit of measurement for this length

        Parameter value: the unit of measurement
        Precondition: value is one of 'in', 'ft', 'yd'
        """
        # enforce invariant
        assert value in ['in', 'ft', 'yd'], repr(value)+' is not a valid unit'
        # assign to attribute
        self._units = value
    
    def getInches(self):
        """
        Returns (a copy of) this fractional length in inches
        """
        n = self.getNumerator()
        d = self.getDenominator()
        if self._units == 'ft':
            n = n*12
        elif self._units == 'yd':
            n = n*36
        
        result = FractionalLength(n,d,'in')
        result.reduce()
        return result
    
    def getFeet(self):
        """
        Returns (a copy of) this fractional length in feet
        """
        n = self.getNumerator()
        d = self.getDenominator()
        if self._units == 'in':
            d = d*12
        elif self._units == 'yd':
            n = n*3
        
        result = FractionalLength(n,d,'in')
        result.reduce()
        return result

    def getYards(self):
        """
        Returns (a copy of) this fractional length in yards
        """
        n = self.getNumerator()
        d = self.getDenominator()
        if self._units == 'in':
            d = d*36
        elif self._units == 'ft':
            d = d*3
        
        result = FractionalLength(n,d,'in')
        result.reduce()
        return result
    
    def __init__(self,n=0,d=1,u='ft'):
        """
        Initializes a new fractional length n/d with units u
        
        Parameter n: the numerator (default is 0)
        Precondition: n is an int (or optional)
        
        Parameter d: the denominator (default is 1)
        Precondition: d is an int > 0 (or optional)
        
        Parameter u: the unit of measurement (default is 'ft')
        Precondition: u is one of 'in', 'ft', 'yd' (or optional)
        """
        super().__init__(n,d)
        self.setUnits(u)
    
    def __str__(self):
        """
        Returns this fractional as a string 'n/d u'
        """
        return super().__str__()+' '+self._units
