"""
Functions on a 2d list of numbers.

This module shows both immutable and mutable functions on nested lists.

Author: Walker M. White (wmw2)
Date:   October 18, 2019
"""
import introcs


# IMMUTABLE
def all_nums(table):
    """
    Returns True if table contains only numbers (False otherwise)
    
    Example: all_nums([[1,2],[3,4]]) is True
    all_nums([[1,2],[3,'a']]) is False
    
    Parameter table: The candidate table to modify
    Preconditions: table is a rectangular 2d List
    """
    result = True
    
    # Walk through table
    for row in table:
    
        # Walk through the row
        for item in row:
            if not type(item) in [int,float]:
                result = False
    
    return result


def rectangular_list(table):
    """
    Returns: True if table is a rectangle 2d list

    Parameter table: The candidate table to check
    Preconditions: None
    """
    if type(table) != list or len(table) == 0:
        return False
    
    # Make sure a table and get the # of columns
    if type(table[0]) != list or len(table[0]) == 0:
        return False
    
    cols = len(table[0])
    for row in table:
        # Make sure table is rectangular
        if type(row) != list or len(row) != cols:
            return False
    
    return True # We made it to the end


def transpose(table):
    """
    Returns: copy of table with rows and columns swapped
    
    Example:
           1  2          1  3  5
           3  4    =>    2  4  6
           5  6
    
    Parameter table: the table to transpose
    Precondition: table is a rectangular 2d List of numbers
    """
    assert rectangular_list(table), repr(table)+' is not a rectangular 2d list'
    assert all_nums(table), repr(table)+' has non-numeric values'
    
    # Find the size of the (non-ragged) table
    numrows = len(table)
    numcols = len(table[0]) # All rows have same no. cols
    
    # Build the table
    result = [] # Result accumulator
    for m in range(numcols):
    
        # Make a single row
        row = [] # Single row accumulator
        for n in range(numrows):
            row.append(table[n][m])
        
        #Add the result to the table
        result.append(row)
    
    return result


def determinant(table):
    """
    Returns: determinant of a 2x2 matrix
    
    Example:
           1  2  =>  1*4 - 3*2 = -2
           3  4
    
    Parameter table: the table to transpose
    Precondition: table is a 2x2 matrix of numbers
    """
    assert rectangular_list(table), repr(table)+' is not a rectangular 2d list'
    assert all_nums(table), repr(table)+' has non-numeric values'
    assert len(table) == 2, 'the table does not have 2 rows'
    assert len(table[0]) == 2, 'the table does not have 2 columns'
    return table[0][0]*table[1][1]-table[0][1]*table[1][0]


# IMMUTABLE
def add_ones(table):
    """
    Adds one to every number in the table
    
    Example: If a= [[1,2],[3,4]]) then add_ones(a) changes a to [[2,3],[4,5]]
    
    Parameter table: The candidate table to modify
    Precondition: table is a rectangular 2d List of numbers
    """
    assert rectangular_list(table), repr(table)+' is not a rectangular 2d list'
    assert all_nums(table), repr(table)+' has non-numeric values'
    
    # Walk through table
    for rpos in range(len(table)):
        
        # Walk through each column
        for cpos in range(len(table[rpos])):
            table[rpos][cpos] = table[rpos][cpos]+1
    
    # No return statement


def strip(table,col):
    """
    Removes column col from the given table
    
    Example: If a= [[1,2,3],[4,5,6]]) then strip(a,1) changes a to [[1,3],[4,6]]
    
    Parameter table: The candidate table to modify
    Precondition: table is a rectangular 2d List of numbers
    
    Parameter col: The column to remove
    Precondition: col is an int and a valid column of table
    """
    assert rectangular_list(table), repr(table)+' is not a rectangular 2d list'
    assert all_nums(table), repr(table)+' has non-numeric values'
    assert type(col) == int, repr(col)+' is not an int'
    assert col >= 0 and col < len(table[0]), repr(col)+' is not a valid column'
    
    # Walk through table
    for rpos in range(len(table)):
        
        # Modify each row to slice out column
        table[rpos] = table[rpos][:col] + table[rpos][col+1:]
    
    # No return statement


# To create a table
def load_table(file):
    """
    Returns: A 2d table of numbers from a CSV file
    
    All numbers will be convered to floats.  Any value that cannot be converted to a 
    float will be treated as a string.
    
    Parameter file: the CSV file to read
    Precondition: file is a CSV file
    """
    table = introcs.read_csv(file)
    for row in range(len(table)):
        for col in range(len(table[0])):
            try:
                value = float(table[row][col])
                table[row][col] = value
            except:
                pass
    return table

