"""
Module showing off how to create a composite object.

A composite object is one that is made up of more than one game2d object, but which can
be moved about on the screen together.

Author: Walker M. White
Date: November 30, 2020
"""
from game2d import *


### Constants ###

# The initial window width
GAME_WIDTH  = 640
# The initial window height
GAME_HEIGHT = 480

# Pixels to move the object each frame
OBJECT_STEP = 2
# The width of the image
IMAGE_WIDTH  = 100
# The height of the image
IMAGE_HEIGHT = 100
# The font size
FONT_SIZE = 64
# How far below the image to put the label
FONT_OFFSET = 40

class App(GameApp):
    """
    The basic app for moving an image with arrow keys

    The image moves so long as the arrow keys are held down. So we do not need to
    distinguish between presses and releases
    """
    # Attribute _comp: The composite object
    # Invariant: _comp is a Combined object

    def start(self):
        """
        Initialize the image
        """
        self._comp = Combined('Click me!',IMAGE_WIDTH,IMAGE_HEIGHT,FONT_SIZE,FONT_OFFSET)
        self._comp.setX(self.width/2)
        self._comp.setY(self.height/2)
        pass # Implement me

    def update(self,dt):
        """
        Move the image about the screen

        Parameter dt: The time since the last animation frame
        Precondition: dt is a float
        """
        pass # Implement me
        if self.input.is_key_down('up'):
            self._comp.setY(self._comp.getY() + OBJECT_STEP)
        if self.input.is_key_down('down'):
            self._comp.setY(self._comp.getY() - OBJECT_STEP)
        if self.input.is_key_down('left'):
            self._comp.setX(self._comp.getX() - OBJECT_STEP)
        if self.input.is_key_down('right'):
            self._comp.setX(self._comp.getX() + OBJECT_STEP)

    def draw(self):
        """
        Draw the image
        """
        self._comp.draw(self.view)
        pass # Implement me


class Combined(object):
    """
    A composite object combining a GImage and GLabel

    Composite objects are created like subcontrollers.  They need a custom
    initializer and custom draw method.  HOWEVER, they are not proper subcontrollers
    because they do not need an update method.  Simply control them with their
    getters and setters
    """
    # Attribute _image: The image object to display
    # Invariant: _image is a GImage object
    #
    # Attribute _label: The text labe to display
    # Invariant: _label is a GLabel object
    #
    # Attribute _x: The x-coordinate of the (averaged) center
    # Invariant: _x is a number and is always the average
    # of _image and _label
    #
    # Attribute _y: The y-coordinate of the (averaged) center
    # Invariant: _y is a number and is always the average of _image and _label
    pass # Implement me

    def getX(self):
        return self._x

    def setX(self,value):
        dx = value - self._x
        self._image.x = self._image.x + dx
        self._label.x = self._label.x + dx
        self._x = value

    def getY(self):
        return self._y

    def setY(self,value):
        dy = value - self._y
        self._image.y = self._image.y + dy
        self._label.y = self._label.y + dy
        self._y = value


    def __init__(self,text,width,height,size,offset):
        """
        Intializes the composite object

        Parameter text: The label text
        Precondition: text is a string

        Parameter width: The image width
        Precondition: width is a number > 0

        Parameter height: The image height
        Precondition: height is a number > 0

        Parameter size: The font size
        Precondition: size is an number > 0

        Parameter offset: The distance from the bottom
        of the image to the center of the label
        Precondition: offset is a number > 0
        """
        self._image = GImage(source='go.png',width=width,height=height)
        self._label = GLabel(text=text,font_size=size)
        self._label.y = self._image.bottom-offset
        self._label.x = self._image.x
        self._x = (self._image.x + self._label.x)/2
        self._y = (self._image.y + self._label.y)/2

    def draw(self,view):
        """
        Draws the composite object

        Paramater view: The view to draw to
        Precondition: view is a GView object
        """
        self._image.draw(view)
        self._label.draw(view)


# Script Code
if __name__ == '__main__':
    App(width=GAME_WIDTH,height=GAME_HEIGHT).run()
