"""
Module showing off how to create a button.

In this demo, we create a button as a subclass of GImage. We switch between a GO and a
STOP sign.

Author: Walker M. White
Date: November 30, 2020
"""
from game2d import *

### Constants ###

# The initial window width
GAME_WIDTH  = 640
# The initial window height
GAME_HEIGHT = 480

# GET RID OF MAGIC NUMBERS
# The width of the button
BUTTON_WIDTH  = 100
# The height of the button
BUTTON_HEIGHT = 100

class App(GameApp):
    """
    The basic app for processing a button click
    """
    # Attribute _button: The button to click on
    # Invariant: _button is a GButton object
    #
    # Attribute _last: The previous touch position
    # Invariant: _last is None or a 2-element tuple
    
    def start(self):
        """
        Initialize the button
        """
        self._button = GButton(BUTTON_WIDTH,BUTTON_HEIGHT)
        self._button.x = self.width//2
        self._button.y = self.height//2
        self._last = None
    
    def update(self,dt):
        """
        Detect a button press
        
        Parameter dt: The time since the last animation frame
        Precondition: dt is a float
        """
        touch = self.input.touch
        
        if touch is None and not self._last is None and self._button.isPressed():
            self._button.setPressed(False)
        elif self._last is None and not touch is None and self._button.contains(touch):
            self._button.setPressed(True)
        
        # For the next time around
        self._last = touch
        
    def draw(self):
        """
        Draw the button
        """
        pass #Implement me
        
        self._button.draw(self.view)


class GButton(GImage):
    """
    A button that changes color when clicked
    
    The button uses the images for GO and STOP to demonstrate if the button is
    pressed.
    """
    # Attribute _pressed: Whether the button is pressed
    # Invariant: _pressed is a bool; if up the image is GO; if down the image is STOP
    
    def isPressed(self):
        """
        Returns True if this button is pressed
        """
        return self._pressed
    
    def setPressed(self,value):
        """
        Sets whether this button is pressed
        
        Parameter value: The new value for pressed
        Precondition: value is a bool
        """
        self._pressed = value
        if value:
            self.source = 'stop.png'
        else:
            self.source = 'go.png'
    
    def __init__(self,width,height):
        """
        Initializes a new button object
        
        Parameter width: The button width
        Precondition: width is a number > 0
        
        Parameter height: The button height
        Precondition: height is a number > 0
        """
        super().__init__(source='go.png',width=width,height=height)
        self._pressed = False


# Script Code
if __name__ == '__main__':
    App(width=GAME_WIDTH,height=GAME_HEIGHT).run()