"""
Module showing off how to create a button.

In this demo, we create a button as a subclass of GLabel.  We add some attributes and
make it switch color when we click on it

Author: Walker M. White
Date: November 30, 2020
"""
from game2d import *

### Constants ###

# The initial window width
GAME_WIDTH  = 640
# The initial window height
GAME_HEIGHT = 480

# REMOVE ME
# Get rid of magic numbers
# The size of the button
FONT_SIZE = 64
# The width of the button
BUTTON_WIDTH  = 300
# The height of the button
BUTTON_HEIGHT = 80
# End remove

class App(GameApp):
    """
    The basic app for processing a button click
    """
    # Attribute _button: The button to click on
    # Invariant: _button is a GButton object
    #
    # Attribute _last: The previous touch position
    # Invariant: _last is None or a 2-element tuple
    
    def start(self):
        """
        Initialize the button
        """
        self._button = GButton('Click me',FONT_SIZE,BUTTON_WIDTH,BUTTON_HEIGHT)
        self._button.x = self.width//2
        self._button.y = self.height//2
        self._last = None
    
    def update(self,dt):
        """
        Detect a button press
        
        Parameter dt: The time since the last animation frame
        Precondition: dt is a float
        """
        touch = self.input.touch
        
        if touch is None and not self._last is None and self._button.isPressed():
            self._button.setPressed(False)
        elif self._last is None and not touch is None and self._button.contains(touch):
            self._button.setPressed(True)
        
        # For the next time around
        self._last = touch
        
    def draw(self):
        """
        Draw the button
        """
        pass #Implement me
        
        self._button.draw(self.view)


### Button Colors ###
# Color when button is up
BUTTON_UP = 'green'
# Color when button is down
BUTTON_DOWN = 'red'

class GButton(GLabel):
    """
    A button that changes color when clicked
    
    The button uses Times New Roman as its font and alternates between
    green and red.
    """
    # Attribute _pressed: Whether the button is pressed
    # Invariant: _pressed is a bool; if up the color is green; if down the color is red
    
    def isPressed(self):
        """
        Returns True if this button is pressed
        """
        return self._pressed
    
    def setPressed(self,value):
        """
        Sets whether this button is pressed
        
        Parameter value: The new value for pressed
        Precondition: value is a bool
        """
        self._pressed = value
        if value:
            self.fillcolor = BUTTON_DOWN
        else:
            self.fillcolor = BUTTON_UP
    
    def __init__(self,text,size,width,height):
        """
        Initializes a new button object
        
        Parameter text: The button text
        Precondition: text is a string
        
        Parameter size: The font size
        Precondition: size is a number > 0
        
        Parameter width: The button width
        Precondition: width is a number > 0
        
        Parameter height: The button height
        Precondition: height is a number > 0
        """
        super().__init__(text=text,font_name='TimesBold.ttf',font_size=size,width=width,height=height)
        self.setPressed(False)


# Script Code
if __name__ == '__main__':
    App(width=GAME_WIDTH,height=GAME_HEIGHT).run()