# ShowPageRank.py
""" Introduction to numpy arrays using 
Page Rank computation as an illustration.

numpy features for 2-dim arrays
     
      Notation:   A[i,j]     is the [i,j] entry in A
                  A[i,:]     1-dim array that is the ith row of A
                  A[:,j]     1-dim array that is the jth row of A
      type:       A = zeros((m,n)) is a mxn array of floats
                  A = zeros((m,n),dtyope='int') is an mxn array of ints
      shape:      (m,n) = A.shape returns row dim in m and col dim in n
      copy:       B = copy(A) makes a copy of the numpy array A
      
numpy features for 1-dim arrays
    
      Notation:   x[i]     is the  i-th entry in A
                  x[i1:i2]
      type:       A = zeros((m,n)) is a mxn array of floats
                  A = zeros((m,n),dtyope='int') is an mxn array of ints
      shape:      (m,n) = A.shape returns row dim in m and col dim in n
      copy:       B = copy(A) makes a copy of the numpy array A
      
      
"""
from numpy import *
from random import uniform as randu

def randuM(m,n,L,U):
    """ Returns an mxn numpy array of random floats
    selected from the interval [L,U]
    
    PreC: m and n are positive ints. L and U are floats
    that  satisfy L<R
    """
    # Use zeros to establish A as an mxn array. Then fill it up
    A = zeros((m,n))
    for i in range(m):
        # Set up row i
        for j in range(n):
            A[i,j] = randu(L,U)
    return A

def randLink(n,p):
    """ Returns an nxn numpy array G of ints that are either
    0 or 1. The probability that an entry is 1 is given by p.
    
    Think: G[i,j] is 1 if there is a link to webpage i from webpage j.
    
    PreC: n is a positive int and p is a float that satisfies 0<p<1.
    """
    # Set p G as an array of ints
    G = zeros((n,n))
    for i in range(n):
        for j in range(n):
            if randu(0,1) <=p:
                G[i,j] = 1
    return G
            

def normalizeCols(A):
    """ Returns a numpy float array B that is obtained by
    dividing each of A's entries by the sum of the
    values in its column. If the sum is zero, then the
    corresponding column of B is ones(m)/m
    
    A is not changed
    
    PreC; A is a numpy array of ints and or floats
    """
    (m,n) = A.shape
    B = copy(A)
    for j in range(n):
        # Get column sum
        sj = sum(B[:,j])
        # and normalize
        if sj==0:
            B[:,j] = ones(m)/m
        else:
            B[:,j] = B[:,j]/sj
    return B
    

def prettyPrint(x):
    n = len(x)
    for k in range(len(x)):
        s = '%6.3f' % x[k]
        print s,
    print
    
def Update(A,x):
    """ Returns a 1-dim numpy array y where for i in range(n)
     
         y[i] = A[i,0]*x[0] + ... + A[i,n-1]*x[n-1]
         
    PreC: A is an nxn numpy array and x is a 1-dim numpy array
    """
    n = len(x)
    y = zeros(n)
    for i in range(n):
        # Compute y[i]
        for j in range(n):
            y[i] += A[i,j]*x[j]
    return y



if __name__ == '__main__':
   # Generate an nxn random link matrix
   n = 10
   prob = .3
   G = randLink(n,prob)
   print '\nThe Link Array:\n'
   print G
   # Generate the transition probabilty transition array
   A = normalizeCols(G)
   # Think: A[i,j] is the proabilty that a websurfer on page j clicks on a
   # link to page i
   # Generate an initial "state array"
   x = randuM(n,1,1,10)
   # Think: There are x[j] random web surferes on page j.
   print
   print 'The initial x and its updates:'
   print
   prettyPrint(x)
   for k in range(2*n):
       x = Update(A,x)
       prettyPrint(x)
    # Observe that the values in x reach a steady state. This provides a
    # measure of a web page's importance. If at the end x[j]>x[i], then this
    # means that web page j is more important than web page i
   

   

