# window.py
# Walker M. White
# August 16, 2012
"""A student-safe wraper for class Tk

The Tkinter libraries appear to be written before Python included modern
classes.  As such they are poorly designed and unsuitable for students in
an introductory course (or, in fact, by anyone, ever).  This module
provides a wrapper class - called Window - that uses modern object-oriented
style."""
from Tkinter import Tk

def _geometry(x,y,width,height):
    """Private function for specifying window geometry"""
    return `width`+"x"+`height`+"+"+`x`+"+"+`y`

class Window(object):
    """Wrapper class for a Tk object"""

    _frame = None # Reference to Tk object

    @property
    def title(self):
        """title of the window, displayed at the top"""
        return self._frame.title()

    @title.setter
    def title(self,value):
        assert type(value) == str,  "%s is not a string" % `value`
        self._frame.title(value)

    @title.deleter
    def title(self):
        self._frame.title(None)

    @property
    def x(self):
        """x coordinate of top left corner of window"""
        return self._frame.winfo_x()

    @x.setter
    def x(self,value):
        assert type(value) == int,  "%s is not an int" % `value`
        geom = _geometry(value,self._frame.winfo_y(),
                         self._frame.winfo_width(),self._frame.winfo_height())
        self._frame.wm_geometry(geom)

    @x.deleter
    def x(self):
        geom = _geometry(0,self._frame.winfo_y(),
                         self._frame.winfo_width(),self._frame.winfo_height())
        self._frame.wm_geometry(geom)

    @property
    def y(self):
        """y coordinate of top left corner of window"""
        return self._frame.winfo_y()

    @y.setter
    def y(self,value):
        assert type(value) == int,  "%s is not an int" % `value`
        geom = _geometry(self._frame.winfo_x(),value,
                         self._frame.winfo_width(),self._frame.winfo_height())
        self._frame.wm_geometry(geom)

    @y.deleter
    def y(self):
        geom = _geometry(self._frame.winfo_x(),0,
                         self._frame.winfo_width(),self._frame.winfo_height())
        self._frame.wm_geometry(geom)

    @property
    def width(self):
        """width of window in pixels"""
        return self._frame.winfo_width()

    @width.setter
    def width(self,value):
        assert type(value) == int,  "%s is not an int" % `value`
        geom = _geometry(self._frame.winfo_x(),self._frame.winfo_y(),
                         value,self._frame.winfo_height())
        self._frame.wm_geometry(geom)

    @width.deleter
    def width(self):
        geom = _geometry(self._frame.winfo_x(),self._frame.winfo_y(),
                         0,self._frame.winfo_height())
        self._frame.wm_geometry(geom)

    @property
    def height(self):
        """height of window in pixels"""
        return self._frame.winfo_height()

    @height.setter
    def height(self,value):
        assert type(value) == int,  "%s is not an int" % `value`
        geom = _geometry(self._frame.winfo_x(),self._frame.winfo_y(),
                         self._frame.winfo_width(),value)
        self._frame.wm_geometry(geom)

    @height.deleter
    def height(self):
        geom = _geometry(self._frame.winfo_x(),self._frame.winfo_y(),
                         self._frame.winfo_width(),0)
        self._frame.wm_geometry(geom)

    @property
    def resizable(self):
        """bool value; True if window can be resized"""
        return self._frame.resizable()[0] == '1'

    @resizable.setter
    def resizable(self,value):
        assert type(value) == bool,  "%s is not a boolean" % `value`
        self._frame.resizable(width=value,height=value)

    @resizable.deleter
    def resizable(self):
        self._frame.resizable(None)

    def __init__(self):
        """Constructor: instances are a window drawn on screen"""
        super(Window,self).__init__()
        self._frame = Tk()

    def __del__(self):
        """Destructor: hide window and delete all graphics assets"""
        self._frame.destroy()

    def beep(self):
        """Play an OS specific alert sound"""
        self._frame.bell()

    def iconify(self):
        """Shrink the window down to an icon, effectively hiding it"""
        self._frame.iconify()

    def deiconify(self):
        """Expand the window from an icon so that it is visible"""
        self._frame.deiconify()

    def setMaxSize(self,width,height):
        """Sets the maximum size for this window
        
        Any attempt to resize a dimension beyond the maximum size
        will fail."""
        self._frame.maxsize(width,height)

    def setMinSize(self,width,height):
        """Sets the minimum size for this window
        
        Any attempt to resize a dimension below the minimum size
        will fail."""
        self._frame.minsize(width,height)