# controller.py
# Walker White (wmw2), Lillian Lee (ljl2), and Steve Marschner (srm2)
# Spring 2013
"""Breakout-like Controller module for lecture demos."""
import colormodel
import random
import math
from graphics import *

WINDOW_WIDTH = 512
WINDOW_HEIGHT = 512

GRAVITY = -0.2

ROCKET_VY = 1.2

PARTICLE_DIAMETER = 5

MAX_INIT_VEL = 5

PARTICLES_PER_SHELL = 50

class Controller(GameController):
    """Instance is a controller for lecture demos about interaction and
    animation.

    This class extends GameController and implements the various methods
    necessary for an interactive application.

        Method initialize is called once at the beginning.

        Method update is called repeatedly to do animation.

        The on_touch methods handle mouse (or finger) input.
    """

    def initialize(self):
        """Initialize the program state."""
        self.view.add(GRectangle(x=0, y=0, size=(WINDOW_WIDTH, WINDOW_HEIGHT)))
        self.particles = []

    def update(self, dt):
        """Animate a single frame.  This is called even when there is
        no user input."""
        # move all the particles
        for particle in self.particles[:]:
            particle.move()
        # delete particles that have fallen out of view
        i = 0
        while i < len(self.particles):
            if self.particles[i].y < -10:
                del self.particles[i]
            else:
                i += 1

    def on_touch_down(self, view, touch):
        """Respond to the mouse (or finger) being pressed (but not released)."""
        rocket = Rocket(self, touch.x, touch.y)
        self.add_particle(rocket)

    def on_touch_move(self, view, touch):
        """Respond to the mouse (or finger) being moved."""
        pass  # IMPLEMENT ME

    def on_touch_up(self, view, touch):
        """Respond to the mouse (or finger) being released."""
        pass  # IMPLEMENT ME

    def add_particle(self, p):
        self.view.add(p)
        self.particles.append(p)

    def del_particle(self, p):        
        self.view.remove(p)
        self.particles.remove(p)

    def explode_shell(self, x, y):
        color = colormodel.RGB(random.randrange(256),random.randrange(256),random.randrange(256))
        for i in range(PARTICLES_PER_SHELL):
            spark = Spark(x, y, color)
            self.add_particle(spark)


class Spark(GEllipse):
    """Instances represent particles created in shell explosions.

    Instance variables:
        vx [float]: velocity in x direction
        vy [float]: velocity in y direction
    """

    def __init__(self, x, y, color=colormodel.WHITE):
        """New particle at (x,y) with random initial velocity and given color."""
        GEllipse.__init__(self, x=x, y=y, size=(PARTICLE_DIAMETER, PARTICLE_DIAMETER),
                          fillcolor=color)        
        self.vy = random.uniform(-MAX_INIT_VEL,MAX_INIT_VEL)
        self.vx = math.sqrt(MAX_INIT_VEL**2 - self.vy**2) * math.sin(random.uniform(0,2*math.pi))

    def move(self):
        self.x += self.vx
        self.y += self.vy
        self.vy += GRAVITY


class Rocket(GEllipse):
    """Instances represent rockets that will generate explosions later.

    Instance variables:
        trigger_y [float]: y coordinate at which rocket will explode
    """

    def __init__(self, controller, x, y):
        """New rocket headed for an explosion at (x,y)."""
        GEllipse.__init__(self, x=x, y=0, size=(PARTICLE_DIAMETER, PARTICLE_DIAMETER),
                          fillcolor=colormodel.GRAY)    
        self.controller = controller
        self.trigger_y = y

    def move(self):
        self.y += ROCKET_VY
        if self.y > self.trigger_y:
            self.controller.explode_shell(self.x, self.y)
            self.controller.del_particle(self)


