# controller.py
# Walker White (wmw2), Lillian Lee (ljl2), and Steve Marschner (srm2)
# Spring 2013
"""Breakout-like Controller module for lecture demos."""
import colormodel
import random
from graphics import *

WINDOW_WIDTH = 512
WINDOW_HEIGHT = 512

STATE_READY = 0
STATE_DRAWING = 1
STATE_MOVING = 2

class Controller(GameController):
    """Instance is a controller for lecture demos about interaction and
    animation.

    This class extends GameController and implements the various methods
    necessary for an interactive application.

    Instance variables:

       shapes [list of GObject]: list containing all the shapes that have been drawn.
       state [int]: which state the program is in
       selected [GObject]: item currently selected, if any; otherwise None
       anchor_offset [pair of float]: offset between mouse and corner of object, when dragging

    Invariants:
        State is one of STATE_READY (ready to draw a shape), STATE_DRAWING (currently
            dragging out a shape), STATE_MOVING (currently moving a shape)
        When mouse is not down, state is STATE_READY
        When mouse is down, state is STATE_DRAWING or STATE_MOVING
        A shape's line color is set to yellow if and only if it is self.selected
    """

    def initialize(self):
        """Initialize the program state."""
        self.shapes = []
        self.state = STATE_READY
        self.selected = None
        self.anchor_offset = None

    def update(self, dt):
        """Animate a single frame.  This is called even when there is
        no user input."""
        pass  # IMPLEMENT ME

    def on_touch_down(self, view, touch):
        """Respond to the mouse (or finger) being pressed (but not released)."""
        # We will be in drawing mode unless we clicked on an object.
        self.state = STATE_DRAWING
        for shape in reversed(self.shapes):
            # If we find an object that we clicked on, select it and set to moving mode
            if shape.collide_point(touch.x, touch.y):
                self.state = STATE_MOVING
                self.anchor_offset = (touch.x - shape.x, touch.y - shape.y)
                self._select_shape(shape)
                break
        # If we are drawing, create a new shape and select it.
        if self.state == STATE_DRAWING:
            new_rect = GRectangle(x=touch.x,y=touch.y,size=(0,0))
            color = colormodel.RGB(random.randrange(256),random.randrange(256),random.randrange(256))
            new_rect.fillcolor = color
            self.view.add(new_rect)
            self.shapes.append(new_rect)
            self._select_shape(new_rect)

    def on_touch_move(self, view, touch):
        """Respond to the mouse (or finger) being moved."""
        if self.state == STATE_DRAWING:
            self.selected.width = touch.x - self.selected.x
            self.selected.height = touch.y - self.selected.y
        if self.state == STATE_MOVING:
            self.selected.x = touch.x - self.anchor_offset[0]
            self.selected.y = touch.y - self.anchor_offset[1]

    def on_touch_up(self, view, touch):
        """Respond to the mouse (or finger) being released."""
        # If the user drew this shape inside out, flip it so that its width and height are
        # both positive.
        if self.state == STATE_DRAWING:
            if self.selected.width < 0:
                self.selected.x += self.selected.width
                self.selected.width = -self.selected.width
            if self.selected.height < 0:
                self.selected.y += self.selected.height
                self.selected.height = -self.selected.height
        # Back to ready state (see invariant)
        self.state = STATE_READY

    # HELPERS

    def _select_shape(self, shape):
        """Make the given shape the selected shape, which means deselecting the 
        currently selected shape, if any."""
        if self.selected is not None:
            self.selected.linecolor = colormodel.BLACK
        self.selected = shape
        if self.selected is not None:
            self.selected.linecolor = colormodel.YELLOW
