# hand.py
# Steve Marschner (srm2) and Lillian Lee (ljl2)
# March 13, 2013
"""Lecture demo: class to represent poker hands.
"""

import card
import random

class Hand(object):
    """Instances represent a hand in poker.
    
    Instance variables:
        cards [list of Card]: cards in the hand
        
    This list is sorted according to the ordering defined by the Card class.
    """
    
    HAND_TYPE_NAMES = ["1 pair", "3 of a kind", "2 pair", "straight", "flush",
                       "full house", "4 of a kind", "straight flush"]
    NUM_HAND_TYPES = len(HAND_TYPE_NAMES)

    def __init__(self, deck=None, n=5, code=''):
        """Draw a hand of n cards.
        Pre: deck is a list of >= n cards.  Deck is shuffled.  *or*
             text is a string of the form '3D 5H KS', consisting of two-
             character codes acceptable to card.Card separated by spaces.
        """
        if code:
            self.cards = map(lambda c: card.Card(code=c), code.split())
        else:
            self.cards = []
            for k in range(n):
                self.cards.append(deck.pop(0))
        self.cards.sort()
        
    def _tally(self):
        """Return: a list of length card.MAX_RANK+2 (call it <counts>) in which
        counts[r] is the number of cards of rank r in the hand."""
        counts = [0] * 15
        for c in self.cards:
            counts[14 if c.rank == 1 else c.rank] += 1
        return counts

    def _tally_single_suit(self, s):
        """Return: a list of length card.MAX_RANK+2 (call it <counts>) in which
        counts[r] is the number of cards of rank r and suit s in the hand."""
        counts = [0] * 15
        for c in self.cards:
            counts[14 if c.rank == 1 else c.rank] += (c.suit == s)
        return counts

    def _tally_suits(self):
        """Return: a list of length card.NUM_SUITS (call it <counts>) in which
        counts[s] is the number of cards of suit s in the hand."""
        counts = [0] * (card.NUM_SUITS)
        for c in self.cards:
            counts[c.suit] += 1
        return counts

    def is_pair(self):
        """Return: This hand contains a pair."""
        return max(self._tally()) >= 2

    def is_2pair(self):
        """Return: This hand contains two distinct pairs."""
        counts = self._tally()
        counts.sort()
        return counts[-1] >= 2 and counts[-2] >= 2
    
    def is_3kind(self):
        """Return: This hand contains three of a kind."""
        return max(self._tally()) >= 3
        
    def is_straight(self):
        """Return: This hand contains a 5-card straight."""
        counts = self._tally()
        return (min(counts[2:7]) > 0 or min(counts[3:8]) > 0 or
                min(counts[4:9]) > 0 or min(counts[5:10]) > 0 or
                min(counts[6:11]) > 0 or min(counts[7:12]) > 0 or
                min(counts[8:13]) > 0 or min(counts[9:14]) > 0 or
                min(counts[10:15]) > 0)
        
    def is_flush(self):
        """Return: This hand contains a flush."""
        return max(self._tally_suits()) >= 5
        
    def is_full_house(self):
        """Return: This hand contains a full house."""
        counts = self._tally()
        counts.sort()
        return counts[-1] >= 3 and counts[-2] >= 2
        
    def is_4kind(self):
        """Return: True if this hand is 4 of a kind."""
        return max(self._tally()) >= 4
    
    def is_straight_flush(self):
        """Return: True if this hand contains a 5-card straight.  That is, if it
        contains five cards that are all in sequence."""
        if self.is_flush() and self.is_straight():
            for suit in range(card.NUM_SUITS):
                counts = self._tally_single_suit(suit)
                if (min(counts[2:7]) > 0 or min(counts[3:8]) > 0 or
                    min(counts[4:9]) > 0 or min(counts[5:10]) > 0 or
                    min(counts[6:11]) > 0 or min(counts[7:12]) > 0 or
                    min(counts[8:13]) > 0 or min(counts[9:14]) > 0 or
                    min(counts[10:15]) > 0):
                    return True
        return False
        
    def get_types(self):
        """Return: a list of booleans indicating whether this hand is or is not
        each of the hand types, indexed like HAND_TYPE_NAMES."""
        return [self.is_pair(), self.is_3kind(), self.is_2pair(),
                self.is_straight(), self.is_flush(), self.is_full_house(),
                self.is_4kind(), self.is_straight_flush()]

    def discard(self, k):
        """Discard the k-th card, counting from zero.
        Precondition: Hand contains at least k+1 cards
        """
        self.cards.remove(k)
        
    def __str__(self):
        return ', '.join(map(str, self.cards))
        
    def __repr__(self):
        return 'Hand(code="' + ' '.join(map(repr, self.cards)) + '")'
        

