# hand.py
# Steve Marschner (srm2) and Lillian Lee (ljl2)
# March 13, 2013
"""Lecture demo: class to represent poker hands.
"""

import card
import random

class Hand(object):
    """Instances represent a hand in poker.
    
    Instance variables:
        cards [list of Card]: cards in the hand
        
    This list is sorted according to the ordering defined by the Card class.
    """
    
    HAND_TYPE_NAMES = ["1 pair", "3 of a kind", "2 pair", "straight", "flush",
                       "full house", "4 of a kind", "straight flush"]
    NUM_HAND_TYPES = len(HAND_TYPE_NAMES)

    def __init__(self, deck=None, n=5, code=None):
        """Draw a hand of n cards.
        Pre: deck is a list of >= n cards.  Deck is shuffled.
            *or*
             code is a string of the form '3D 5H KS', consisting of two-
             character codes acceptable to card.Card separated by spaces.
        """
        self.cards = []
        if code is not None:
            for card_code in code.split():
                self.cards.append(card.Card(code=card_code))
        else:
            for k in range(n):
                self.cards.append(deck.pop(0))
        self.cards.sort()
        
    def _tally_ranks(self):
        """make list of counts"""
        counts = [0] * (card.MAX_RANK+1)
        for c in self.cards:
            counts[c.rank] += 1
        return counts
        
    def is_pair(self):
        """Return: This hand contains a pair."""
        return max(self._tally_ranks()) >= 2

    def is_2pair(self):
        """Return: This hand contains two distinct pairs."""
        counts = self._tally_ranks()
        counts.sort()
        return counts[-2] >= 2
    
    def is_3kind(self):
        """Return: This hand contains three of a kind."""
        return max(self._tally_ranks()) >= 3
        
    def is_straight(self):
        """Return: This hand contains five cards that are all in sequence."""
        return False
                
    def is_flush(self):
        """Return: This hand contains five cards of the same suit."""
        c = self.cards
        return c[0].suit == c[1].suit == c[2].suit == c[3].suit == c[4].suit
        
    def is_full_house(self):
        """Return: This hand contains three of a kind and a distinct pair."""
        return False
        
    def is_4kind(self):
        """Return: This hand contains four of a kind."""
        c = self.cards
        return c[0].rank == c[3].rank or c[1].rank == c[4].rank
        
    def is_straight_flush(self):
        """Return: This hand contains 5 cards of the same suit that are all
        in sequence."""
        return False
        
    def discard(self, k):
        """Discard the k-th card, counting from zero.
        Precondition: Hand contains at least k+1 cards
        """
        self.cards.remove(k)
        
    def __str__(self):
        return ', '.join(map(str, self.cards))



























#Extra stuff...

#    def get_types(self):
#        """Return: a list of booleans indicating whether this hand is or is not
#        each of the hand types, indexed like HAND_TYPE_NAMES."""
#        return [self.is_pair(), self.is_3kind(), self.is_2pair(),
#                self.is_straight(), self.is_flush(), self.is_full_house(),
#                self.is_4kind(), self.is_straight_flush()]
"""                
            *or*
             text is a string of the form '3D 5H KS', consisting of two-
             character codes acceptable to card.Card separated by spaces.

"""