import java.awt.*;

import java.net.*;
import java.awt.event.*;
import javax.swing.*;
import java.io.File;
import java.io.IOException;


/** An instance is a JFrame that contains an image (in two forms:
  the  original image and a current version)  with a title and
  buttons to alter the current image.
  When no image is in the JFrame, it is hidden.
  When an image is placed in it, it becomes visible. */
public class ImageGUI extends JFrame implements ActionListener  {
    
    // contains the original and current ImageArrays and
    // methods that manipulate them.
    private ImageProcessor processor; 
    
    // Labels and list for the JButtons to be put in buttonBox
    private final String[] buttonLabelList= {"restore", "invert", "transpose",
        "hor reflect", "ver reflect", "filter out", "hide message", 
        "reveal message", "fuzzify", "put in jail", "save image"};
    private JButton[] buttonList = new JButton[buttonLabelList.length];
    private Box buttonBox= new Box(BoxLayout.Y_AXIS);
    
    // Three radio buttons
    private JRadioButton greyButton= new JRadioButton("all color", true);
    private JRadioButton redButton= new JRadioButton("red", false);
    private JRadioButton greenButton= new JRadioButton("green", false);
    private JRadioButton blueButton= new JRadioButton("blue", false);
    private Box checkboxBox= new Box(BoxLayout.X_AXIS);
    
    private ButtonGroup group= new ButtonGroup();
    
    
    /** The panel with the original image, the panel with the current image,
      and the box that contains both. */
    private ImagePanel originalPanel;
    private ImagePanel currentPanel;
    Box inputBox= new Box(BoxLayout.Y_AXIS);
    
    // The text area for the user to give a message
    private JTextArea messageArea= new JTextArea(10, 20);
    private JScrollPane scrollPane= new JScrollPane(messageArea);
    private JLabel jlabel= new JLabel("Type the message to be hidden in this text area:");
    Box areaBox= new Box(BoxLayout.Y_AXIS);
    
    // imageBox contains the original image and the current one
    Box imagebox= new Box(BoxLayout.X_AXIS);
    
    /** Constructor: an instance for Image im.
      If im is null, the window is not shown. */
    public ImageGUI(Image im) {
        setUp(im, "An image");
    }
    
    
    /** Constructor: an instance for an Image that is found using a file dialog
      window. If no Image is produced using the file dialog,
      use null as the Image. */
    public ImageGUI() {
        String f= getImageName();
        Image im= getImage(f, this);
        setUp(im, "image: " + f); 
    }
    
    /** = the current panel */
    public ImagePanel getCurrentPanel() {
        return currentPanel;
    }
    
    /** = the original panel */
    public ImagePanel getOriginalPanel() {
        return originalPanel;
    }
    
    /** = the ImageProcessor for the image. */
    public ImageProcessor getProcessor() {
        return processor;
    }
    
    
    /** Set up this JFrame for Image im, with title t. This includes creating the
      ImageProcessor, adding buttons and checkboxes to the GUI, 
      performing a few other minor operations to make the GUI
      work properly, and placing the image in the JFrame.*/
    private void setUp(Image im, String t) {
        setTitle(t);
        
        // Build box buttonBox of buttons.
        
        /* inv: buttonList[0..i-1] contain JButtons with labels
         * buttonLabelList[0..i-1], each JButton has been added to buttonBox,
         *  and this ImageGUI is registered as a listener to these JButtons
         */        
        for (int i=0; i < buttonLabelList.length; i= i+1) {
            buttonList[i]= new JButton(buttonLabelList[i]);
            buttonBox.add(buttonList[i]);
            buttonList[i].addActionListener(this);
        }      
        getContentPane().add(BorderLayout.WEST, buttonBox);
        
        // Build box inputBox of the color buttons.
        checkboxBox.add(greyButton);
        checkboxBox.add(redButton);
        checkboxBox.add(greenButton);
        checkboxBox.add(blueButton);
        inputBox.add(checkboxBox);
        
        // Add the color buttons to group.
        group.add(greyButton);
        group.add(redButton);
        group.add(blueButton);
        group.add(greenButton);
        
        getContentPane().add(BorderLayout.NORTH, inputBox);
        
        // Add a text area with label
        areaBox.add(jlabel);
        areaBox.add(scrollPane);
        getContentPane().add(BorderLayout.SOUTH, areaBox);
        
        // If there is no image, make JFrame invisible and return
        if (im == null) {
            setVisible(false);
            return;
        }
        
        // Create the two panels and the ImageProcessor processor.
        originalPanel= new ImagePanel(im);
        currentPanel= new ImagePanel(im);
        
        // Add the panels to the GUI, in imagebox.
        imagebox= new Box(BoxLayout.X_AXIS);
        imagebox.add(originalPanel);
        imagebox.add(currentPanel);
        getContentPane().add(BorderLayout.EAST, imagebox);
        
        // Create an instance of ImageProcessor to manipulate the image.
        ImageArray intIm= new ImageArray(im,
                                         im.getHeight(originalPanel), im.getWidth(originalPanel));
        processor= new ImageProcessor(intIm);
        
        
        setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
        setLocation(200,100);
        pack();
        setVisible(true);
        repaint();
    }
    
    /** Pause the program for d microseconds */
    public void pause(int d) {
        try { Thread.currentThread().sleep(d); }
        catch (InterruptedException e) { }
    }
    
    /** = the grey check box in the GUI is selected */
    private boolean greyIsChecked() {
        return greyButton.isSelected();
    }
    /** = the red check box in the GUI is selected */
    private boolean redIsChecked() {
        return redButton.isSelected();
    }
    
    /** = the green check box in the GUI is selected */
    private boolean greenIsChecked() {
        return greenButton.isSelected();
    }
    
    /** = the blue check box in the GUI is selected */
    private boolean blueIsChecked() {
        return blueButton.isSelected();
    }
    
    /** Change current image to the map given by 
      processor.getCurrentImage(). */
    private void changeCurrentImage() {
        currentPanel.changeImageTo(processor.getCurrentImage());
        pack();
        repaint();
    }
    
    /** = the file name of an image loaded by the user using a dialog window */
    public static String getImageName() {
        FileDialog fd = new FileDialog(new Frame(), "Open File", FileDialog.LOAD);
        fd.setVisible(true);
        if (fd.getFile() == null) 
            return null;
        return fd.getDirectory() + fd.getFile();
    }
    
    /** = the image for file name f, using the toolkit for jframe */
    public static Image getImage(String f, JFrame jframe) {
        Image image= null;
        try {
            image= jframe.getToolkit().getImage(new URL("file:" + f));
            
        } 
        catch (MalformedURLException e) {
            System.err.println("Bad URL!");
            return null;
        }
        
        // set media tracker to wait for image to load
        MediaTracker tracker= new MediaTracker(jframe);
        tracker.addImage(image,0);    
        
        // wait for image to load
        try { tracker.waitForID(0); 
        }
        catch (InterruptedException e) {
            // handler.flashMessage(ImageHandler.LOAD_INTERRUPTED);
            return null;
        }
        return image;
    }
    
    /** Process a click of one of the buttons */
    public void actionPerformed(ActionEvent e)  {
        // assert statements are simply reminders about which function to call
        // (since buttonLabelList is "far away" in this code)
        
        if (e.getSource() == buttonList[0]) {
            assert buttonLabelList[0].equals("restore");
            processor.restore();
            changeCurrentImage();
            return;
        }
        
        if (e.getSource() == buttonList[1]) {
            assert buttonLabelList[1].equals("invert");
            processor.invert();
            changeCurrentImage();
            return;
        }
        
        if (e.getSource() == buttonList[2]) {
            assert buttonLabelList[2].equals("transpose"); 
            processor.transpose();
            changeCurrentImage();
            return;
        }
        
        if (e.getSource() == buttonList[3]) {
            assert buttonLabelList[3].equals("hor reflect"); 
            processor.hreflect();
            changeCurrentImage();
            return;
        }
        
        if (e.getSource() == buttonList[4]) {
            assert buttonLabelList[4].equals("ver reflect"); 
            processor.vreflect();
            changeCurrentImage();
            return;
        }
        
        if (e.getSource() == buttonList[5]) {
            assert buttonLabelList[5].equals("filter out");
            if (greyIsChecked()) {
                processor.filterOut(ImageProcessor.GRAY);
                changeCurrentImage();
                return;
            }
            if (redIsChecked()) {
                processor.filterOut(ImageProcessor.RED);
                changeCurrentImage();
                return;
            }
            if (greenIsChecked()) {
                processor.filterOut(ImageProcessor.GREEN);
                changeCurrentImage();
                return;
            }
            if (blueIsChecked()) {
                processor.filterOut(ImageProcessor.BLUE);
                changeCurrentImage();
                return;
            }
        }
        
        if (e.getSource() == buttonList[6]) {
            assert buttonLabelList[6].equals("hide message");
            
            String m= messageArea.getText();
            boolean b= processor.hide(m);
            if (b) {
                messageArea.setText("Message of length " + m.length() + " was hidden.\n");
                changeCurrentImage();
            } else {
                messageArea.setText("Message too long to be hidden:\n" + m);
            }
            
            return;
        }
        
        if (e.getSource() == buttonList[7]) {
            assert buttonLabelList[7].equals("reveal message");        
            String m= processor.reveal();
            if (m == null) {
                messageArea.setText("No message found to reveal.");
            } else {
                messageArea.setText(m);
            }
            return;
        }
        
        if (e.getSource() == buttonList[8]) {
            assert buttonLabelList[8].equals("fuzzify");
            processor.fuzzify();
            changeCurrentImage();
            return; 
            
        }
        
        if (e.getSource() == buttonList[9]) {
            assert buttonLabelList[9].equals("put in jail");       
            processor.putInJail();
            changeCurrentImage();
            return; 
            
        }
        
        if (e.getSource() == buttonList[10]) {
            assert buttonLabelList[10].equals("save image");       
            try {
                JFileChooser fd = new JFileChooser("Save to file");
                fd.showSaveDialog(null);
                File f= fd.getSelectedFile();
                if (f == null) {
                    System.out.println("Failed to get a file name");
                    return;
                }
                
                String fileName= f.getAbsolutePath();
                
                processor.writeImage(fileName);
            }
            catch (IOException ex) { 
                System.out.println("Write failed somehow.");
            }
            changeCurrentImage();
            return; 
            
        }      
    }
}
