// Assume In.java has been downloaded to this directory

/** An instance holds a text with lie spans delimited */
public class LieData {
    
    /** lie spans in text are demarcated by matching spanStart/spanEnd pairs.
      * text is all lowercase, with no embedded spans.
      * spanStart and spanEnd must not be super- or substrings of each other, 
      * and must be at least one character long (once initialized). 
      */
    private String text; 
    private String spanStart;
    private String spanEnd;
    
    /** Constructor: a LieFile whose text's lies are assumed to be delimited by 
      * <font color="red"> and </font>.
      * Preconditions: source is the name of a file containing all lower-case
      * text and containing at least one character but no embedded spans.
      */
    public LieData(String sourceFile) {
        text= (new In(sourceFile)).readAll();  
        spanStart= "<font color=\"red\">";
        spanEnd= "</font>";
    }
    
    /** Constructor: a LieFile whose text's lies are delimited by
      * matching spanStart and spanEnd pairs.
      * Preconditions: source is the name of a containing all lower-case
      * text and containing at least one character but no embedded spans.
      * spanStart and spanEnd must not be super- or substrings of each other, 
      * and must be at least one character long. 
      */
    public LieData(String sourceFile, String spanStart, String spanEnd) {
        text= (new In(sourceFile)).readAll();
        this.spanStart= spanStart;
        this.spanEnd= spanEnd;
    }
   
    /** = "w at index iW in text is a hit (in a span)"
      * Precondition:  iW is an index of an instance of w in text */
    public boolean inLieSpan(String w, int iW) {
     
        // Assertion: iW starts a w
        
        // get index of closest spanEnd to the right of w
        int iEnd= text.indexOf(spanEnd, iW+w.length());
        
        // = "there is such a spanEnd and its matching spanStart
        // is before iW, so iW is in a span."
        return iEnd != -1 && 
            text.lastIndexOf(spanStart, iEnd) < iW; 

    }
    
      
    /* = a TallyPair whose hits (respectively misses) are the number of
     * occurrences of w in text within a span (respectively not within a span)
     * starting at iW (if iW is -1, the TallyPair will have 0 hits and misses),
     * given that 
     * iW = index of current occurrence of w in text (-1 if none)    
     * Preconditions: none of w, spanStart, and spanEnd are super/substrings of one of the others;
     * w contains at least one character and is all lowercase;
     */
    // Note that this method is private
    private TallyPair remainingCounts(String w, int iW) {
        
        if (iW == -1) {
            return new TallyPair();
        }
        
        // get counts for part of text after w at iW
        TallyPair p= remainingCounts(w,text.indexOf(w,iW+w.length()));
        if (inLieSpan(w,iW)) {
            p.hits= p.hits+1;
        } else {
            p.misses= p.misses+1;
        }
        return p;
        
    }
       
    /** = a String of the form "hits: h; misses: m", where h (respectively m)
      * is the number of occurrences of w inside (respectively outside) delimited spans
      * in the text this LieFile stores.
      * Precondition: w is not a superstring or substring of either span delimiter, 
      * and contains at least one character, and is all lowercase
      */
    public String counts(String w) {
         return remainingCounts(w,text.indexOf(w)).toString();
    }
   
        

}