
import java.awt.*;
import java.awt.image.*;

/* An instance is an image map: an internal version of an image. See
   the constructor to see how an image is "grabbed" and stored for
   manipulation. */
public class ImageMap {
    /** Provides methods for extracting components of an rgb pixdel.*/
    public final static DirectColorModel DM= (DirectColorModel) ColorModel.getRGBdefault();
    
    Image image; // The image
    int rows;    // number of rows in the image
    int cols;    // number of columns in the image
    int[] map;   // the pixel map
    
    /** = the number of rows */
    public int getRows() {
        return rows;
    }
    
    /** = the number of columns */
    public int getCols() {
        return cols;
    }
    
    /** = the map */
    public int[] getMap() {
        return map;
    }
    
    /** = the image associated with this map. */
    public Image getImage() {
     return image;   
    }
    
    /** Set number of rows to r */
    public void setRows(int r) {
        rows= r;
    }
    
    /** Set number of columns to c */
    public void setCols(int c) {
        cols= c;
    }
    
    /** Constructor: A map for image v with r rows and c cols.
        A variable of class Image contains a .jpeg or .gif image.
        Class PixelGrabber is given an image, as well as the number
        of rows and cols it should contain, and an int array map into
        which this image should be stored.
       
       Calling method grabPixels then causes the image to be stored
       in one-dimensional array map --even though a picture is a
       two-dimensional thing. The two-dimensional array of elements
       is stored in map in row-major order. */
    public ImageMap(Image im, int r, int c) {
        image= im;
        rows= r;
        cols= c;
        map= new int[r*c];
        if (im == null) return;
        PixelGrabber pg=
             new PixelGrabber(im, 0, 0, c, r, map, 0, c);
        try {
            pg.grabPixels();
        }
        catch (InterruptedException e) {
            System.out.println("pixel grab interrupted!");
            return;
        } 
    }
    
    /** = a copy of this image map. */
    public ImageMap copy() {
        return new ImageMap(image,rows,cols);
    }
    
    /** = the pixel value in the image map at [row,col]. */
    public int getPixel(int row, int col) {
        return map[row*cols + col];
    }
    
    /** set the pixel value in the image map at [row,col] to v. */
    public void setPixel(int row, int col, int v) {
        map[row*cols + col]= v;
    }
    
    /** swap the pixel at [a,b] with the pixel at [i,j]. */
    public void swapPixels(int a, int b, int i, int j) {
        int temp= getPixel(a,b);   // temporary pixel
        setPixel(a,b,getPixel(i,j));
        setPixel(i,j,temp);
    }
    
    /** = pixel number p of the map, in row major order,
          with pixel number 0 being the first. */
    public int getPixel(int p) {
        return map[p];
    }
    
    /** Set pixel number p of the map (in row major order) to v. */
    public void setPixel(int p, int v) {
        map[p]= v;
    }
    
    /** = a pixel value pix, in the form (red, green, blue) */
    public static String toString (int pix) {
        int red= DM.getRed(pix);
        int green=  DM.getGreen(pix);
        int blue=  DM.getBlue(pix);
        int alpha=  DM.getAlpha(pix);
        
        return "(" + red + ", " + green + ", " + blue + ")";
    }
}

